"""
API client for Paplix/Gitea repository management.

Handles repository CRUD operations via the Paplix web app API.
"""

import json
import ssl
from dataclasses import dataclass
from typing import List, Optional
from urllib.request import Request, urlopen
from urllib.error import URLError, HTTPError
from urllib.parse import urlencode


# Module-level variable to track SSL warning
_ssl_warning: Optional[str] = None


def get_ssl_warning() -> Optional[str]:
    """Get the SSL warning message if SSL verification was disabled."""
    return _ssl_warning


def clear_ssl_warning():
    """Clear the SSL warning."""
    global _ssl_warning
    _ssl_warning = None


def _create_ssl_context(verified: bool = True):
    """Create SSL context.

    Args:
        verified: If True, create verified context. If False, create unverified.
    """
    if not verified:
        ctx = ssl.create_default_context()
        ctx.check_hostname = False
        ctx.verify_mode = ssl.CERT_NONE
        return ctx

    # Try certifi package first (if installed)
    try:
        import certifi
        return ssl.create_default_context(cafile=certifi.where())
    except ImportError:
        pass

    # Try default context with system certs
    return ssl.create_default_context()


def _urlopen_with_ssl_fallback(request: Request, timeout: int = 30):
    """
    Open URL with SSL fallback.

    First tries with SSL verification, falls back to unverified if SSL error.
    Sets module-level _ssl_warning if fallback was used.

    Returns:
        Response object
    """
    global _ssl_warning

    # First try with SSL verification
    try:
        return urlopen(request, context=_create_ssl_context(verified=True), timeout=timeout)
    except URLError as e:
        error_str = str(e)
        # Check if it's an SSL-related error
        ssl_indicators = ['SSL', 'ssl', 'certificate', 'CERTIFICATE', 'TLS', 'tls']
        if any(indicator in error_str for indicator in ssl_indicators):
            # Retry without SSL verification
            _ssl_warning = ("SSL certificate verification was disabled. "
                           "The connection may not be secure.")
            return urlopen(request, context=_create_ssl_context(verified=False), timeout=timeout)
        raise


class APIError(Exception):
    """Exception raised for API errors."""
    pass


class AuthenticationError(APIError):
    """Exception raised for authentication errors."""
    pass


class RepositoryExistsError(APIError):
    """Exception raised when repository already exists."""
    pass


@dataclass
class Repository:
    """Represents a Gitea repository."""
    id: int
    name: str
    full_name: str
    description: str
    private: bool
    empty: bool
    clone_url: str
    ssh_url: str
    html_url: str
    default_branch: str
    created_at: str
    updated_at: Optional[str] = None
    clone_url_authenticated: Optional[str] = None

    @classmethod
    def from_dict(cls, data: dict) -> 'Repository':
        """Create Repository from API response dict."""
        return cls(
            id=data.get('id', 0),
            name=data.get('name', ''),
            full_name=data.get('full_name', ''),
            description=data.get('description', ''),
            private=data.get('private', True),
            empty=data.get('empty', True),
            clone_url=data.get('clone_url', ''),
            ssh_url=data.get('ssh_url', ''),
            html_url=data.get('html_url', ''),
            default_branch=data.get('default_branch', 'main'),
            created_at=data.get('created_at', ''),
            updated_at=data.get('updated_at'),
            clone_url_authenticated=data.get('clone_url_authenticated')
        )

    def get_authenticated_url(self, gitea_token: Optional[str] = None) -> str:
        """
        Get the authenticated clone URL.

        Uses clone_url_authenticated if available from API,
        otherwise builds it from clone_url and provided token.

        Args:
            gitea_token: Gitea access token (used if clone_url_authenticated not set)

        Returns:
            Authenticated clone URL with token embedded
        """
        if self.clone_url_authenticated:
            return self.clone_url_authenticated

        if gitea_token and self.clone_url:
            # Build authenticated URL: http://token@host/path
            if '://' in self.clone_url:
                protocol, rest = self.clone_url.split('://', 1)
                return f"{protocol}://{gitea_token}@{rest}"

        return self.clone_url


class PaplixAPIClient:
    """
    API client for Paplix web app.

    Handles repository management via the Paplix API.
    """

    def __init__(self, base_url: str, oauth_token: str):
        """
        Initialize API client.

        Args:
            base_url: Base URL of the Paplix web app (e.g., "http://localhost:3000")
            oauth_token: OAuth access token from authentication
        """
        self.base_url = base_url.rstrip('/')
        self.oauth_token = oauth_token

    @property
    def _headers(self) -> dict:
        """Get request headers with authentication."""
        return {
            'Authorization': f'Bearer {self.oauth_token}',
            'Content-Type': 'application/json',
            'Accept': 'application/json'
        }

    def _request(self, method: str, endpoint: str, data: Optional[dict] = None) -> dict:
        """
        Make an API request.

        Args:
            method: HTTP method (GET, POST, DELETE)
            endpoint: API endpoint path
            data: Optional request body data

        Returns:
            Response data as dictionary

        Raises:
            APIError: If request fails
            AuthenticationError: If authentication fails
            RepositoryExistsError: If repository already exists (409)
        """
        url = f"{self.base_url}{endpoint}"

        try:
            request = Request(url, method=method, headers=self._headers)

            if data is not None:
                request.data = json.dumps(data).encode('utf-8')

            with _urlopen_with_ssl_fallback(request, timeout=30) as response:
                response_text = response.read().decode('utf-8')
                if response_text:
                    return json.loads(response_text)
                return {}

        except HTTPError as e:
            error_body = ''
            try:
                error_body = e.read().decode('utf-8')
                error_data = json.loads(error_body)
                error_message = error_data.get('error', error_body)
            except (json.JSONDecodeError, AttributeError):
                error_message = error_body or str(e)

            if e.code == 401:
                raise AuthenticationError("Invalid or expired access token")
            elif e.code == 409:
                raise RepositoryExistsError(f"Repository already exists: {error_message}")
            else:
                raise APIError(f"API request failed ({e.code}): {error_message}")

        except URLError as e:
            raise APIError(f"Network error: {e.reason}")
        except json.JSONDecodeError as e:
            raise APIError(f"Invalid API response: {e}")

    def create_repository(
        self,
        name: str,
        description: Optional[str] = None,
        private: bool = True,
        auto_init: bool = False,
        default_branch: str = "main"
    ) -> Repository:
        """
        Create a new Gitea repository.

        Args:
            name: Repository name (alphanumeric, hyphens, underscores)
            description: Optional repository description
            private: Whether repository should be private (default: True)
            auto_init: Initialize with README (default: False)
            default_branch: Default branch name (default: "main")

        Returns:
            Repository object with clone URL and other info

        Raises:
            RepositoryExistsError: If repository already exists
            AuthenticationError: If authentication fails
            APIError: If creation fails
        """
        payload = {
            'name': name,
            'private': private,
            'auto_init': auto_init,
            'default_branch': default_branch
        }

        if description:
            payload['description'] = description

        response = self._request('POST', '/api/gitea/repositories', payload)

        if response.get('success') and 'repository' in response:
            return Repository.from_dict(response['repository'])
        else:
            raise APIError("Unexpected response format")

    def list_repositories(self) -> List[Repository]:
        """
        List all repositories for the authenticated user.

        Returns:
            List of Repository objects
        """
        response = self._request('GET', '/api/gitea/repositories')

        repositories = []
        for repo_data in response.get('repositories', []):
            repositories.append(Repository.from_dict(repo_data))

        return repositories

    def get_repository(self, name: str) -> Optional[Repository]:
        """
        Get a specific repository by name.

        Args:
            name: Repository name

        Returns:
            Repository object if found, None otherwise
        """
        repositories = self.list_repositories()
        for repo in repositories:
            if repo.name == name:
                return repo
        return None

    def repository_exists(self, name: str) -> bool:
        """
        Check if a repository exists.

        Args:
            name: Repository name

        Returns:
            True if repository exists
        """
        return self.get_repository(name) is not None

    def get_authenticated_clone_url(self, clone_url: str, gitea_token: str) -> str:
        """
        Get clone URL with embedded authentication token.

        Args:
            clone_url: Original clone URL (e.g., http://localhost:3001/org/repo.git)
            gitea_token: Gitea access token

        Returns:
            Clone URL with embedded token (e.g., http://token@localhost:3001/org/repo.git)
        """
        # Insert token after protocol
        if '://' in clone_url:
            protocol, rest = clone_url.split('://', 1)
            return f"{protocol}://{gitea_token}@{rest}"
        return clone_url
