"""
Smart change detection system for git repository monitoring.

Provides efficient change detection using:
- File system watching (via watchdog library)
- State caching to avoid redundant git calls
- Background thread monitoring
- Graceful fallback to polling if watching fails
"""

import os
import time
import threading
from pathlib import Path
from dataclasses import dataclass, field
from typing import Optional, Dict, List, Set, Callable, Any, Tuple

import wx
import wx.lib.newevent

# Custom events for change notification
GitStateChangedEvent, EVT_GIT_STATE_CHANGED = wx.lib.newevent.NewEvent()


@dataclass
class GitStateSnapshot:
    """Immutable snapshot of git repository state."""
    head_hash: str = ""
    branch: str = ""
    staged_files: List[str] = field(default_factory=list)
    modified_files: List[str] = field(default_factory=list)
    untracked_files: List[str] = field(default_factory=list)
    ahead: int = 0
    behind: int = 0
    has_remote: bool = False
    is_merging: bool = False
    timestamp: float = field(default_factory=time.time)

    @property
    def staged_count(self) -> int:
        return len(self.staged_files)

    @property
    def modified_count(self) -> int:
        return len(self.modified_files)

    @property
    def untracked_count(self) -> int:
        return len(self.untracked_files)

    @property
    def total_changes(self) -> int:
        return self.staged_count + self.modified_count + self.untracked_count

    @property
    def has_changes(self) -> bool:
        return self.total_changes > 0

    def differs_from(self, other: Optional['GitStateSnapshot']) -> bool:
        """Check if this snapshot differs from another."""
        if other is None:
            return True
        return (
            self.head_hash != other.head_hash or
            self.branch != other.branch or
            self.staged_files != other.staged_files or
            self.modified_files != other.modified_files or
            self.untracked_files != other.untracked_files or
            self.ahead != other.ahead or
            self.behind != other.behind or
            self.has_remote != other.has_remote or
            self.is_merging != other.is_merging
        )

    def get_changes(self, other: Optional['GitStateSnapshot']) -> Dict[str, Any]:
        """Return dict describing what changed compared to another snapshot."""
        if other is None:
            return {'full_refresh': True}

        changes = {}

        if self.branch != other.branch:
            changes['branch'] = self.branch

        if self.head_hash != other.head_hash:
            changes['head'] = self.head_hash

        if (self.staged_files != other.staged_files or
            self.modified_files != other.modified_files or
            self.untracked_files != other.untracked_files):
            changes['files'] = {
                'staged': self.staged_files,
                'modified': self.modified_files,
                'untracked': self.untracked_files
            }

        if self.ahead != other.ahead or self.behind != other.behind:
            changes['ahead_behind'] = (self.ahead, self.behind)

        if self.has_remote != other.has_remote:
            changes['remote'] = self.has_remote

        if self.is_merging != other.is_merging:
            changes['merging'] = self.is_merging

        return changes


class GitStateCache:
    """Thread-safe cache for git state to avoid redundant calls."""

    def __init__(self, git):
        """
        Initialize the cache.

        Args:
            git: GitUtils instance for running git commands
        """
        self._git = git
        self._lock = threading.RLock()
        self._snapshot: Optional[GitStateSnapshot] = None
        self._invalidated = True

    def get_snapshot(self) -> Optional[GitStateSnapshot]:
        """Get the current cached snapshot."""
        with self._lock:
            return self._snapshot

    def invalidate(self):
        """Force cache to be refreshed on next access."""
        with self._lock:
            self._invalidated = True

    def refresh(self, force: bool = False) -> Tuple[bool, Optional[GitStateSnapshot]]:
        """
        Refresh cache if needed.

        Args:
            force: Force refresh even if cache is valid

        Returns:
            Tuple of (changed, new_snapshot)
        """
        with self._lock:
            if not force and not self._invalidated:
                return False, self._snapshot

            try:
                new_snapshot = self._fetch_state()
            except Exception:
                return False, self._snapshot

            old_snapshot = self._snapshot
            changed = new_snapshot.differs_from(old_snapshot)

            self._snapshot = new_snapshot
            self._invalidated = False

            return changed, new_snapshot

    def _fetch_state(self) -> GitStateSnapshot:
        """Fetch current git state."""
        # Get status using existing GitUtils method
        status = self._git.get_status()

        # Get HEAD hash
        head_hash = self._git.get_head_hash_quick() or ""

        # Check if merging
        is_merging = self._git.is_merging()

        return GitStateSnapshot(
            head_hash=head_hash,
            branch=status.branch or "",
            staged_files=list(status.staged),
            modified_files=list(status.modified),
            untracked_files=list(status.untracked),
            ahead=status.ahead,
            behind=status.behind,
            has_remote=status.has_remote,
            is_merging=is_merging,
            timestamp=time.time()
        )


class GitFileSystemWatcher:
    """
    Watches .git directory for changes using watchdog library.

    Monitors:
    - HEAD (branch switches, commits)
    - refs/heads/* (branch creation/deletion)
    - index (staging area changes)
    - MERGE_HEAD (merge state)
    - FETCH_HEAD (fetch completed)
    """

    DEBOUNCE_SECONDS = 0.3

    def __init__(self, repo_path: str, callback: Callable[[], None]):
        """
        Initialize the watcher.

        Args:
            repo_path: Path to the git repository
            callback: Function to call when changes detected
        """
        self._repo_path = repo_path
        self._git_dir = Path(repo_path) / '.git'
        self._callback = callback
        self._observer = None
        self._debounce_timer: Optional[threading.Timer] = None
        self._lock = threading.Lock()
        self._running = False

    def start(self) -> bool:
        """
        Start watching.

        Returns:
            True if started successfully, False if watchdog unavailable
        """
        try:
            from watchdog.observers import Observer
            from watchdog.events import FileSystemEventHandler

            class GitEventHandler(FileSystemEventHandler):
                def __init__(handler_self, watcher):
                    handler_self.watcher = watcher

                def on_any_event(handler_self, event):
                    # Filter to relevant git files
                    path = event.src_path
                    if any(x in path for x in ['HEAD', 'refs', 'index', 'MERGE_HEAD', 'FETCH_HEAD', 'objects']):
                        handler_self.watcher._on_change()

            if not self._git_dir.exists():
                return False

            self._observer = Observer()
            handler = GitEventHandler(self)
            self._observer.schedule(handler, str(self._git_dir), recursive=True)
            self._observer.start()
            self._running = True
            return True

        except ImportError:
            # watchdog not available
            return False
        except Exception:
            return False

    def stop(self):
        """Stop watching and cleanup."""
        self._running = False
        with self._lock:
            if self._debounce_timer:
                self._debounce_timer.cancel()
                self._debounce_timer = None

        if self._observer:
            self._observer.stop()
            self._observer.join(timeout=2.0)
            self._observer = None

    def _on_change(self):
        """Handle file system event with debouncing."""
        with self._lock:
            # Cancel existing timer
            if self._debounce_timer:
                self._debounce_timer.cancel()

            # Schedule new callback after debounce period
            self._debounce_timer = threading.Timer(
                self.DEBOUNCE_SECONDS,
                self._trigger_callback
            )
            self._debounce_timer.start()

    def _trigger_callback(self):
        """Trigger the callback after debounce."""
        if self._running:
            self._callback()


class ChangeDetectionManager:
    """
    Coordinates change detection with fallback strategies.

    Strategy hierarchy:
    1. File system watching (preferred - instant detection)
    2. Smart polling with caching (fallback)
    """

    POLL_INTERVAL_SECONDS = 5.0

    def __init__(self, parent: wx.Window, project_path: str, git):
        """
        Initialize the manager.

        Args:
            parent: Parent wx.Window for posting events
            project_path: Path to the git repository
            git: GitUtils instance
        """
        self._parent = parent
        self._project_path = project_path
        self._git = git

        # Components
        self._cache = GitStateCache(git)
        self._watcher: Optional[GitFileSystemWatcher] = None
        self._poll_thread: Optional[threading.Thread] = None
        self._stop_event = threading.Event()

        # State
        self._watching = False
        self._started = False

    def start(self):
        """Start change detection."""
        if self._started:
            return

        self._started = True
        self._stop_event.clear()

        # Try file system watching first
        self._watcher = GitFileSystemWatcher(
            self._project_path,
            self._on_fs_change
        )

        if self._watcher.start():
            self._watching = True
        else:
            self._watching = False

        # Always run polling as backup (less frequently if watching)
        self._start_polling()

        # Do initial check
        self._check_for_changes(force=True)

    def stop(self):
        """Stop all monitoring."""
        if not self._started:
            return

        self._started = False
        self._stop_event.set()

        if self._watcher:
            self._watcher.stop()
            self._watcher = None

        if self._poll_thread:
            self._poll_thread.join(timeout=2.0)
            self._poll_thread = None

    def force_refresh(self):
        """Force immediate refresh (user-triggered)."""
        self._cache.invalidate()
        self._check_for_changes(force=True)

    def _on_fs_change(self):
        """Handle file system watcher event."""
        self._cache.invalidate()
        # Run check in background thread
        threading.Thread(
            target=self._check_for_changes,
            daemon=True
        ).start()

    def _start_polling(self):
        """Start polling in background thread."""
        self._poll_thread = threading.Thread(
            target=self._poll_loop,
            daemon=True
        )
        self._poll_thread.start()

    def _poll_loop(self):
        """Background polling loop."""
        # Use longer interval if watching is active
        interval = self.POLL_INTERVAL_SECONDS * 2 if self._watching else self.POLL_INTERVAL_SECONDS

        while not self._stop_event.wait(timeout=interval):
            if not self._started:
                break
            self._cache.invalidate()
            self._check_for_changes()

    def _check_for_changes(self, force: bool = False):
        """
        Check for changes using cached state comparison.
        Posts wx events if changes detected.
        """
        if not self._started:
            return

        try:
            changed, snapshot = self._cache.refresh(force=force)

            if changed and snapshot:
                old_snapshot = self._cache.get_snapshot()
                changes = snapshot.get_changes(old_snapshot) if old_snapshot else {'full_refresh': True}

                # Post event to main thread
                evt = GitStateChangedEvent(
                    snapshot=snapshot,
                    changes=changes
                )
                wx.PostEvent(self._parent, evt)

        except Exception:
            # Silently ignore errors in background thread
            pass
