"""
Configuration management for Paplix Version Control Plugin.

Handles persistent storage of:
- Base URL for the Paplix web service
- OAuth access tokens and authentication data
"""

import json
import os
from pathlib import Path
from typing import Optional


class Config:
    """Manages plugin configuration storage and retrieval."""

    CONFIG_FILENAME = "paplix_vc_config.json"

    def __init__(self):
        self._config_dir = self._get_config_dir()
        self._config_path = self._config_dir / self.CONFIG_FILENAME
        self._data = self._load()

    def _get_config_dir(self) -> Path:
        """Get the configuration directory path based on the platform."""
        if os.name == 'nt':  # Windows
            config_dir = Path(os.environ.get('APPDATA', '')) / 'kicad' / 'paplix_vc'
        elif os.name == 'posix':
            if 'darwin' in os.uname().sysname.lower():  # macOS
                config_dir = Path.home() / 'Library' / 'Application Support' / 'kicad' / 'paplix_vc'
            else:  # Linux
                config_dir = Path.home() / '.config' / 'kicad' / 'paplix_vc'
        else:
            config_dir = Path.home() / '.paplix_vc'

        config_dir.mkdir(parents=True, exist_ok=True)
        return config_dir

    def _load(self) -> dict:
        """Load configuration from file."""
        if self._config_path.exists():
            try:
                with open(self._config_path, 'r', encoding='utf-8') as f:
                    return json.load(f)
            except (json.JSONDecodeError, IOError):
                return {}
        return {}

    def _save(self) -> None:
        """Save configuration to file."""
        try:
            with open(self._config_path, 'w', encoding='utf-8') as f:
                json.dump(self._data, f, indent=2)
        except IOError as e:
            raise ConfigError(f"Failed to save configuration: {e}")

    @property
    def base_url(self) -> Optional[str]:
        """Get the configured base URL for the Paplix service."""
        return self._data.get('base_url')

    @base_url.setter
    def base_url(self, value: str) -> None:
        """Set the base URL for the Paplix service."""
        # Normalize URL (remove trailing slash)
        if value:
            value = value.rstrip('/')
        self._data['base_url'] = value
        self._save()

    @property
    def access_token(self) -> Optional[str]:
        """Get the stored OAuth access token."""
        return self._data.get('access_token')

    @access_token.setter
    def access_token(self, value: str) -> None:
        """Store the OAuth access token."""
        self._data['access_token'] = value
        self._save()

    @property
    def refresh_token(self) -> Optional[str]:
        """Get the stored OAuth refresh token."""
        return self._data.get('refresh_token')

    @refresh_token.setter
    def refresh_token(self, value: str) -> None:
        """Store the OAuth refresh token."""
        self._data['refresh_token'] = value
        self._save()

    @property
    def user_info(self) -> Optional[dict]:
        """Get stored user information."""
        return self._data.get('user_info')

    @user_info.setter
    def user_info(self, value: dict) -> None:
        """Store user information."""
        self._data['user_info'] = value
        self._save()

    @property
    def token_expires_at(self) -> Optional[int]:
        """Get token expiration timestamp."""
        return self._data.get('token_expires_at')

    @token_expires_at.setter
    def token_expires_at(self, value: int) -> None:
        """Store token expiration timestamp."""
        self._data['token_expires_at'] = value
        self._save()

    @property
    def gitea_url(self) -> Optional[str]:
        """Get the Gitea URL for git operations."""
        return self._data.get('gitea_url')

    @gitea_url.setter
    def gitea_url(self, value: str) -> None:
        """Store the Gitea URL."""
        self._data['gitea_url'] = value
        self._save()

    @property
    def gitea_access_token(self) -> Optional[str]:
        """Get the Gitea access token for git operations."""
        return self._data.get('gitea_access_token')

    @gitea_access_token.setter
    def gitea_access_token(self, value: str) -> None:
        """Store the Gitea access token."""
        self._data['gitea_access_token'] = value
        self._save()

    @property
    def gitea_username(self) -> Optional[str]:
        """Get the Gitea username."""
        return self._data.get('gitea_username')

    @gitea_username.setter
    def gitea_username(self, value: str) -> None:
        """Store the Gitea username."""
        self._data['gitea_username'] = value
        self._save()

    @property
    def gitea_org_name(self) -> Optional[str]:
        """Get the Gitea organization name."""
        return self._data.get('gitea_org_name')

    @gitea_org_name.setter
    def gitea_org_name(self, value: str) -> None:
        """Store the Gitea organization name."""
        self._data['gitea_org_name'] = value
        self._save()

    def is_authenticated(self) -> bool:
        """Check if user is authenticated with valid credentials."""
        return bool(self.gitea_access_token and self.gitea_url and self.base_url)

    def clear_auth(self) -> None:
        """Clear all authentication data."""
        keys_to_remove = [
            'access_token', 'refresh_token', 'user_info', 'token_expires_at',
            'gitea_url', 'gitea_access_token', 'gitea_username', 'gitea_org_name'
        ]
        for key in keys_to_remove:
            self._data.pop(key, None)
        self._save()

    def clear_all(self) -> None:
        """Clear all configuration data."""
        self._data = {}
        self._save()


class ConfigError(Exception):
    """Exception raised for configuration errors."""
    pass


class ProjectConfig:
    """Manages project-specific configuration stored in the project directory."""

    CONFIG_FILENAME = ".paplix_config.json"

    # Default values for project settings
    DEFAULTS = {
        'auto_commit_on_focus': False,
        'auto_push_on_focus': False,
        'auto_pull_on_open': False,
        'default_commit_message': 'Auto-save from KiCad',
        'default_branch': 'master',
        # Panel state
        'panel_expanded': False,
        'panel_x': -1,
        'panel_y': -1,
        'panel_width': 400,
        'panel_height': 500,
    }

    def __init__(self, project_path: str):
        """
        Initialize project configuration.

        Args:
            project_path: Path to the project directory
        """
        self._project_dir = Path(project_path)
        self._config_path = self._project_dir / self.CONFIG_FILENAME
        self._data = self._load()

    def _load(self) -> dict:
        """Load configuration from project directory."""
        if self._config_path.exists():
            try:
                with open(self._config_path, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                    # Merge with defaults to ensure all keys exist
                    return {**self.DEFAULTS, **data}
            except (json.JSONDecodeError, IOError):
                return dict(self.DEFAULTS)
        return dict(self.DEFAULTS)

    def _save(self) -> None:
        """Save configuration to project directory."""
        try:
            with open(self._config_path, 'w', encoding='utf-8') as f:
                json.dump(self._data, f, indent=2)
        except IOError as e:
            raise ConfigError(f"Failed to save project configuration: {e}")

    @property
    def auto_commit_on_focus(self) -> bool:
        """Auto-commit when plugin opens."""
        return self._data.get('auto_commit_on_focus', False)

    @auto_commit_on_focus.setter
    def auto_commit_on_focus(self, value: bool) -> None:
        self._data['auto_commit_on_focus'] = value
        self._save()

    @property
    def auto_push_on_focus(self) -> bool:
        """Auto-push after auto-commit."""
        return self._data.get('auto_push_on_focus', False)

    @auto_push_on_focus.setter
    def auto_push_on_focus(self, value: bool) -> None:
        self._data['auto_push_on_focus'] = value
        self._save()

    @property
    def auto_pull_on_open(self) -> bool:
        """Auto-pull when plugin opens."""
        return self._data.get('auto_pull_on_open', False)

    @auto_pull_on_open.setter
    def auto_pull_on_open(self, value: bool) -> None:
        self._data['auto_pull_on_open'] = value
        self._save()

    @property
    def default_commit_message(self) -> str:
        """Default message for auto-commits."""
        return self._data.get('default_commit_message', 'Auto-save from KiCad')

    @default_commit_message.setter
    def default_commit_message(self, value: str) -> None:
        self._data['default_commit_message'] = value
        self._save()

    @property
    def default_branch(self) -> str:
        """Default/preferred branch name."""
        return self._data.get('default_branch', 'master')

    @default_branch.setter
    def default_branch(self, value: str) -> None:
        self._data['default_branch'] = value
        self._save()

    def save_all(self, auto_commit_on_focus: bool, auto_push_on_focus: bool,
                 auto_pull_on_open: bool, default_commit_message: str,
                 default_branch: str) -> None:
        """Save all settings at once (more efficient than individual saves)."""
        self._data['auto_commit_on_focus'] = auto_commit_on_focus
        self._data['auto_push_on_focus'] = auto_push_on_focus
        self._data['auto_pull_on_open'] = auto_pull_on_open
        self._data['default_commit_message'] = default_commit_message
        self._data['default_branch'] = default_branch
        self._save()

    # Panel state properties
    @property
    def panel_expanded(self) -> bool:
        """Whether the panel is in expanded state."""
        return self._data.get('panel_expanded', False)

    @panel_expanded.setter
    def panel_expanded(self, value: bool) -> None:
        self._data['panel_expanded'] = value
        self._save()

    @property
    def panel_x(self) -> int:
        """Panel X position."""
        return self._data.get('panel_x', -1)

    @property
    def panel_y(self) -> int:
        """Panel Y position."""
        return self._data.get('panel_y', -1)

    @property
    def panel_width(self) -> int:
        """Panel width."""
        return self._data.get('panel_width', 400)

    @property
    def panel_height(self) -> int:
        """Panel height."""
        return self._data.get('panel_height', 500)

    def save_panel_state(self, expanded: bool, x: int, y: int,
                         width: int, height: int) -> None:
        """Save panel position and size."""
        self._data['panel_expanded'] = expanded
        self._data['panel_x'] = x
        self._data['panel_y'] = y
        self._data['panel_width'] = width
        self._data['panel_height'] = height
        self._save()
