"""
wxPython dialogs for Paplix Version Control Plugin.

Provides the user interface for all plugin operations including:
- Setup and configuration
- Git operations (commit, push, pull, branch management)
- OAuth authentication
"""

import wx
import wx.lib.newevent
import os
import re
import sys
import threading
from typing import Optional, List, Callable

import pcbnew

from .config import Config, ProjectConfig
from .git_utils import GitUtils, GitStatus, GitError, GitNotFoundError, GitOperationResult, MergeResult, RebaseResult
from .git_graph import GitGraphPanel
from .change_detector import ChangeDetectionManager, EVT_GIT_STATE_CHANGED
from .oauth import OAuthHandler, OAuthTokens, OAuthError, UserInfo, get_ssl_warning as get_oauth_ssl_warning, clear_ssl_warning as clear_oauth_ssl_warning
from .api_client import PaplixAPIClient, Repository, APIError, AuthenticationError, RepositoryExistsError, get_ssl_warning as get_api_ssl_warning, clear_ssl_warning as clear_api_ssl_warning


# Status message types
STATUS_SUCCESS = "success"
STATUS_WARNING = "warning"
STATUS_ERROR = "error"
STATUS_INFO = "info"


# Custom events for async operations
AuthCompleteEvent, EVT_AUTH_COMPLETE = wx.lib.newevent.NewEvent()
AuthErrorEvent, EVT_AUTH_ERROR = wx.lib.newevent.NewEvent()
GitOperationCompleteEvent, EVT_GIT_OP_COMPLETE = wx.lib.newevent.NewEvent()
GitOperationErrorEvent, EVT_GIT_OP_ERROR = wx.lib.newevent.NewEvent()


class VersionControlPanel(wx.Frame):
    """
    Non-modal floating panel for version control operations.

    Features two states:
    - Collapsed: Compact mini-bar showing status at a glance
    - Expanded: Full panel with all controls
    """

    # Dimensions
    COLLAPSED_WIDTH = 280
    COLLAPSED_HEIGHT = 36
    DEFAULT_EXPANDED_WIDTH = 400
    DEFAULT_EXPANDED_HEIGHT = 500
    MIN_EXPANDED_WIDTH = 350
    MIN_EXPANDED_HEIGHT = 400

    def __init__(self, parent, project_path: str, config: Config):
        # Start with collapsed size, will expand if needed
        super().__init__(
            parent,
            title="Paplix VC",
            style=wx.FRAME_TOOL_WINDOW | wx.FRAME_FLOAT_ON_PARENT | wx.RESIZE_BORDER | wx.CAPTION | wx.CLOSE_BOX
        )

        self.project_path = project_path
        self.config = config
        self.project_config = ProjectConfig(project_path)
        self.git = GitUtils(project_path)

        self.is_expanded = False
        self._expanded_size = wx.Size(
            self.project_config.panel_width or self.DEFAULT_EXPANDED_WIDTH,
            self.project_config.panel_height or self.DEFAULT_EXPANDED_HEIGHT
        )

        # Auto-refresh timer (3 seconds while focused) - fallback for change detection
        self._window_active = False
        self.refresh_timer = wx.Timer(self)

        # Smart change detection system
        self._change_detector = ChangeDetectionManager(
            parent=self,
            project_path=project_path,
            git=self.git
        )

        # Create both UIs (we'll show/hide as needed)
        self._create_ui()

        # Restore position if saved
        self._restore_position()

        # Start in saved state (collapsed by default)
        if self.project_config.panel_expanded:
            self._expand()
        else:
            self._collapse()

        # Bind events
        self.Bind(wx.EVT_CLOSE, self._on_close)
        self.Bind(wx.EVT_SIZE, self._on_resize)
        self.Bind(wx.EVT_MOVE_END, self._on_move_end)

        # Bind custom auth events
        self.Bind(EVT_AUTH_COMPLETE, self._on_auth_complete)
        self.Bind(EVT_AUTH_ERROR, self._on_auth_error)

        # Bind auto-refresh events
        self.Bind(wx.EVT_ACTIVATE, self._on_activate)
        self.Bind(wx.EVT_TIMER, self._on_timer_tick, self.refresh_timer)

        # Bind smart change detection event
        self.Bind(EVT_GIT_STATE_CHANGED, self._on_git_state_changed)

        # Start change detection and initial refresh
        self._change_detector.start()
        self._refresh_status()

    def _create_ui(self):
        """Create both collapsed and expanded UI panels."""
        self.main_panel = wx.Panel(self)
        self.main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Collapsed UI (mini-bar)
        self.collapsed_panel = self._create_collapsed_ui()
        self.main_sizer.Add(self.collapsed_panel, 0, wx.EXPAND)

        # Expanded UI (full panel)
        self.expanded_panel = self._create_expanded_ui()
        self.main_sizer.Add(self.expanded_panel, 1, wx.EXPAND)

        self.main_panel.SetSizer(self.main_sizer)

    def _create_collapsed_ui(self) -> wx.Panel:
        """Create the compact mini-bar UI."""
        panel = wx.Panel(self.main_panel)
        panel.SetBackgroundColour(wx.Colour(45, 45, 48))  # Dark background

        sizer = wx.BoxSizer(wx.HORIZONTAL)
        sizer.AddSpacer(8)

        # Branch indicator (colored dot)
        self.branch_dot = wx.StaticText(panel, label="●")
        self.branch_dot.SetForegroundColour(wx.Colour(34, 197, 94))  # Green
        self.branch_dot.SetFont(wx.Font(12, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL))
        sizer.Add(self.branch_dot, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(4)

        # Branch name
        self.branch_label = wx.StaticText(panel, label="main")
        self.branch_label.SetForegroundColour(wx.WHITE)
        self.branch_label.SetFont(wx.Font(10, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        sizer.Add(self.branch_label, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(12)

        # Ahead/behind indicator
        self.sync_label = wx.StaticText(panel, label="")
        self.sync_label.SetForegroundColour(wx.Colour(200, 200, 200))
        self.sync_label.SetFont(wx.Font(9, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL))
        sizer.Add(self.sync_label, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(8)

        # Changes count
        self.changes_label = wx.StaticText(panel, label="")
        self.changes_label.SetForegroundColour(wx.Colour(234, 179, 8))  # Yellow
        self.changes_label.SetFont(wx.Font(9, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL))
        sizer.Add(self.changes_label, 0, wx.ALIGN_CENTER_VERTICAL)

        sizer.AddStretchSpacer()

        # Expand button
        self.expand_btn = wx.Button(panel, label="▲", size=(28, 24))
        self.expand_btn.SetBackgroundColour(wx.Colour(60, 60, 65))
        self.expand_btn.SetForegroundColour(wx.WHITE)
        self.expand_btn.Bind(wx.EVT_BUTTON, self._on_toggle_expand)
        self.expand_btn.SetToolTip("Expand panel")
        sizer.Add(self.expand_btn, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(4)

        panel.SetSizer(sizer)

        # Double-click to expand
        panel.Bind(wx.EVT_LEFT_DCLICK, self._on_toggle_expand)
        self.branch_label.Bind(wx.EVT_LEFT_DCLICK, self._on_toggle_expand)

        return panel

    def _create_expanded_ui(self) -> wx.Panel:
        """Create the full expanded panel UI."""
        panel = wx.Panel(self.main_panel)
        sizer = wx.BoxSizer(wx.VERTICAL)

        # Header bar
        header = self._create_header(panel)
        sizer.Add(header, 0, wx.EXPAND)

        # Commit section (message input + buttons) - at top
        commit_section = self._create_commit_section(panel)
        sizer.Add(commit_section, 0, wx.EXPAND | wx.ALL, 8)

        # Status section (changes list) - below commit section, max height
        self.changes_list_panel = self._create_changes_list(panel)
        sizer.Add(self.changes_list_panel, 0, wx.EXPAND | wx.LEFT | wx.RIGHT, 8)
        self.changes_list_panel.SetMinSize((-1, 180))  # Force minimum height

        # Git graph (history) section
        self.history_panel = self._create_history_section(panel)
        sizer.Add(self.history_panel, 1, wx.EXPAND | wx.LEFT | wx.RIGHT | wx.BOTTOM, 8)

        # Status bar
        self.status_text = wx.StaticText(panel, label="")
        self.status_text.SetFont(wx.Font(8, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL))
        sizer.Add(self.status_text, 0, wx.EXPAND | wx.LEFT | wx.RIGHT | wx.BOTTOM, 8)

        panel.SetSizer(sizer)
        return panel

    def _create_header(self, parent) -> wx.Panel:
        """Create the compact header with branch and sync controls."""
        panel = wx.Panel(parent)
        panel.SetBackgroundColour(wx.Colour(45, 45, 48))

        sizer = wx.BoxSizer(wx.HORIZONTAL)
        sizer.AddSpacer(8)

        # Branch indicator
        self.header_branch_dot = wx.StaticText(panel, label="●")
        self.header_branch_dot.SetForegroundColour(wx.Colour(34, 197, 94))
        self.header_branch_dot.SetFont(wx.Font(12, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL))
        sizer.Add(self.header_branch_dot, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(4)

        # Branch name
        self.header_branch_label = wx.StaticText(panel, label="main")
        self.header_branch_label.SetForegroundColour(wx.WHITE)
        self.header_branch_label.SetFont(wx.Font(10, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        sizer.Add(self.header_branch_label, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(12)

        # Sync status
        self.header_sync_label = wx.StaticText(panel, label="")
        self.header_sync_label.SetForegroundColour(wx.Colour(200, 200, 200))
        sizer.Add(self.header_sync_label, 0, wx.ALIGN_CENTER_VERTICAL)

        sizer.AddStretchSpacer()

        # Action buttons
        btn_size = (28, 24)
        btn_bg = wx.Colour(60, 60, 65)

        # Refresh
        self.refresh_btn = wx.Button(panel, label="⟳", size=btn_size)
        self.refresh_btn.SetBackgroundColour(btn_bg)
        self.refresh_btn.SetForegroundColour(wx.WHITE)
        self.refresh_btn.Bind(wx.EVT_BUTTON, self._on_refresh)
        self.refresh_btn.SetToolTip("Refresh status")
        sizer.Add(self.refresh_btn, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(2)

        # Pull
        self.pull_btn = wx.Button(panel, label="↓", size=btn_size)
        self.pull_btn.SetBackgroundColour(btn_bg)
        self.pull_btn.SetForegroundColour(wx.WHITE)
        self.pull_btn.Bind(wx.EVT_BUTTON, self._on_pull)
        self.pull_btn.SetToolTip("Pull from remote")
        sizer.Add(self.pull_btn, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(2)

        # Push
        self.push_btn = wx.Button(panel, label="↑", size=btn_size)
        self.push_btn.SetBackgroundColour(btn_bg)
        self.push_btn.SetForegroundColour(wx.WHITE)
        self.push_btn.Bind(wx.EVT_BUTTON, self._on_push)
        self.push_btn.SetToolTip("Push to remote")
        sizer.Add(self.push_btn, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(2)

        # Settings menu
        self.settings_btn = wx.Button(panel, label="⋮", size=btn_size)
        self.settings_btn.SetBackgroundColour(btn_bg)
        self.settings_btn.SetForegroundColour(wx.WHITE)
        self.settings_btn.Bind(wx.EVT_BUTTON, self._on_settings_menu)
        self.settings_btn.SetToolTip("Settings")
        sizer.Add(self.settings_btn, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(4)

        # Collapse button
        self.collapse_btn = wx.Button(panel, label="▼", size=btn_size)
        self.collapse_btn.SetBackgroundColour(btn_bg)
        self.collapse_btn.SetForegroundColour(wx.WHITE)
        self.collapse_btn.Bind(wx.EVT_BUTTON, self._on_toggle_expand)
        self.collapse_btn.SetToolTip("Collapse to mini-bar")
        sizer.Add(self.collapse_btn, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.AddSpacer(4)

        panel.SetSizer(sizer)
        panel.SetMinSize((-1, 36))
        return panel

    def _create_changes_list(self, parent) -> wx.Panel:
        """Create the changes list panel with max height."""
        panel = wx.Panel(parent)
        sizer = wx.BoxSizer(wx.VERTICAL)

        # Header
        header_sizer = wx.BoxSizer(wx.HORIZONTAL)
        self.changes_header = wx.StaticText(panel, label="CHANGES (0)")
        self.changes_header.SetFont(wx.Font(9, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        header_sizer.Add(self.changes_header, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.Add(header_sizer, 0, wx.EXPAND | wx.BOTTOM, 4)

        # File list with max height (300px)
        self.files_list = wx.ListCtrl(panel, style=wx.LC_REPORT | wx.LC_NO_HEADER | wx.LC_SINGLE_SEL)
        self.files_list.InsertColumn(0, "Status", width=30)
        self.files_list.InsertColumn(1, "File", width=300)
        self.files_list.SetMinSize((-1, 150))
        self.files_list.SetMaxSize((-1, 300))
        sizer.Add(self.files_list, 1, wx.EXPAND)

        panel.SetSizer(sizer)
        panel.SetMaxSize((-1, 330))  # Header + list max
        return panel

    def _create_history_section(self, parent) -> wx.Panel:
        """Create the git history/graph section."""
        panel = wx.Panel(parent)
        sizer = wx.BoxSizer(wx.VERTICAL)

        # Header
        header_sizer = wx.BoxSizer(wx.HORIZONTAL)
        self.history_header = wx.StaticText(panel, label="HISTORY")
        self.history_header.SetFont(wx.Font(9, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        header_sizer.Add(self.history_header, 0, wx.ALIGN_CENTER_VERTICAL)
        sizer.Add(header_sizer, 0, wx.EXPAND | wx.TOP | wx.BOTTOM, 8)

        # Git graph panel
        self.git_graph = GitGraphPanel(panel, self.git)
        self.git_graph.SetMinSize((-1, 100))
        sizer.Add(self.git_graph, 1, wx.EXPAND)

        panel.SetSizer(sizer)
        return panel

    def _create_commit_section(self, parent) -> wx.Panel:
        """Create the commit message and buttons section."""
        panel = wx.Panel(parent)
        sizer = wx.BoxSizer(wx.VERTICAL)

        # Commit message input
        self.commit_input = wx.TextCtrl(panel, style=wx.TE_PROCESS_ENTER | wx.TE_MULTILINE, size=(-1, 60))
        self.commit_input.SetHint("Commit message...")
        self.commit_input.Bind(wx.EVT_TEXT_ENTER, self._on_commit)
        sizer.Add(self.commit_input, 0, wx.EXPAND | wx.BOTTOM, 4)

        # Commit All button (full width)
        self.commit_btn = wx.Button(panel, label="✓ Commit All")
        self.commit_btn.Bind(wx.EVT_BUTTON, self._on_commit)
        sizer.Add(self.commit_btn, 0, wx.EXPAND | wx.BOTTOM, 4)

        # Push button (full width, shown only when there are unpushed commits)
        self.section_push_btn = wx.Button(panel, label="↑ Push")
        self.section_push_btn.Bind(wx.EVT_BUTTON, self._on_push)
        sizer.Add(self.section_push_btn, 0, wx.EXPAND)
        self.section_push_btn.Hide()  # Hidden by default

        panel.SetSizer(sizer)
        return panel

    def _collapse(self):
        """Collapse to mini-bar."""
        self.is_expanded = False
        self.expanded_panel.Hide()
        self.collapsed_panel.Show()
        self.SetSize((self.COLLAPSED_WIDTH, self.COLLAPSED_HEIGHT))
        self.SetMinSize((self.COLLAPSED_WIDTH, self.COLLAPSED_HEIGHT))
        self.SetMaxSize((self.COLLAPSED_WIDTH, self.COLLAPSED_HEIGHT))
        self.expand_btn.SetLabel("▲")
        self.main_panel.Layout()
        self._save_state()

    def _expand(self):
        """Expand to full panel."""
        self.is_expanded = True
        self.collapsed_panel.Hide()
        self.expanded_panel.Show()
        self.SetMinSize((self.MIN_EXPANDED_WIDTH, self.MIN_EXPANDED_HEIGHT))
        self.SetMaxSize((-1, -1))
        self.SetSize(self._expanded_size)
        self.collapse_btn.SetLabel("▼")
        self.main_panel.Layout()
        self._save_state()

    def _on_toggle_expand(self, event):
        """Toggle between collapsed and expanded state."""
        if self.is_expanded:
            self._collapse()
        else:
            self._expand()

    def _restore_position(self):
        """Restore saved window position."""
        x = self.project_config.panel_x
        y = self.project_config.panel_y
        if x >= 0 and y >= 0:
            # Validate position is on screen
            display = wx.Display()
            rect = display.GetClientArea()
            if x < rect.width and y < rect.height:
                self.SetPosition((x, y))
            else:
                self.Centre()
        else:
            self.Centre()

    def _save_state(self):
        """Save current panel state."""
        pos = self.GetPosition()
        size = self.GetSize() if self.is_expanded else self._expanded_size
        self.project_config.save_panel_state(
            expanded=self.is_expanded,
            x=pos.x,
            y=pos.y,
            width=size.width,
            height=size.height
        )

    def _on_close(self, event):
        """Handle window close - hide instead of destroy."""
        # Stop auto-refresh timer
        if self.refresh_timer.IsRunning():
            self.refresh_timer.Stop()
        # Stop change detection
        self._change_detector.stop()
        self._save_state()
        self.Hide()

    def _on_resize(self, event):
        """Handle resize - save expanded size."""
        if self.is_expanded:
            self._expanded_size = self.GetSize()
        event.Skip()

    def _on_move_end(self, event):
        """Handle window move - save position."""
        self._save_state()
        event.Skip()

    def _on_activate(self, event):
        """Handle window activation/deactivation for auto-refresh."""
        self._window_active = event.GetActive()
        if self._window_active:
            # Start change detection when focused
            self._change_detector.start()
            # Force immediate refresh
            self._change_detector.force_refresh()
            # Start backup timer (3 seconds)
            if not self.refresh_timer.IsRunning():
                self.refresh_timer.Start(3000)
        else:
            # Stop timer when window loses focus
            if self.refresh_timer.IsRunning():
                self.refresh_timer.Stop()
            # Stop change detection to save resources
            self._change_detector.stop()
        event.Skip()

    def _on_timer_tick(self, event):
        """Handle periodic refresh while window is active (backup for change detection)."""
        if self._window_active:
            # Force refresh via change detector
            self._change_detector.force_refresh()

    def _on_git_state_changed(self, event):
        """Handle git state change from change detection system."""
        snapshot = event.snapshot
        changes = event.changes

        # Smart update - only refresh what changed
        if 'full_refresh' in changes or not changes:
            # Full refresh needed
            self._refresh_status()
            return

        # Update only changed elements
        if 'branch' in changes:
            self.branch_label.SetLabel(snapshot.branch or "No branch")
            self.header_branch_label.SetLabel(snapshot.branch or "No branch")

        if 'files' in changes:
            self._update_changes_list_from_snapshot(snapshot)

        if 'ahead_behind' in changes:
            if snapshot.behind > 0:
                self.header_status_label.SetLabel(f"↓ {snapshot.behind}")
                self.header_status_label.SetForegroundColour(wx.Colour(234, 179, 8))
            else:
                self.header_status_label.SetLabel("")

        if 'head' in changes:
            # New commits - refresh git graph
            if hasattr(self, 'git_graph'):
                self.git_graph.refresh()

    def _update_changes_list_from_snapshot(self, snapshot):
        """Update the changes list from a state snapshot."""
        self.changes_list.Clear()

        # Combine all files with status prefixes
        for f in snapshot.staged_files:
            self.changes_list.Append(f"[A] {f}")
        for f in snapshot.modified_files:
            self.changes_list.Append(f"[M] {f}")
        for f in snapshot.untracked_files:
            self.changes_list.Append(f"[?] {f}")

        # Update counter
        total = snapshot.total_changes
        self.changes_count.SetLabel(str(total) if total > 0 else "")

        # Update branch color based on changes
        if total > 0:
            self.branch_label.SetForegroundColour(wx.Colour(234, 179, 8))
        else:
            self.branch_label.SetForegroundColour(wx.Colour(34, 197, 94))

    def _refresh_status(self):
        """Refresh git status and update UI."""
        status = self.git.get_status()

        # Update branch display
        branch = status.branch or "No branch"
        self.branch_label.SetLabel(branch)
        self.header_branch_label.SetLabel(branch)

        # Update branch color based on status
        if status.has_changes:
            color = wx.Colour(234, 179, 8)  # Yellow - uncommitted changes
        else:
            color = wx.Colour(34, 197, 94)  # Green - clean
        self.branch_dot.SetForegroundColour(color)
        self.header_branch_dot.SetForegroundColour(color)

        # Update sync status (only show behind, ahead count is in push button)
        sync_text = ""
        if status.behind > 0:
            sync_text = f"↓{status.behind}"
        self.sync_label.SetLabel(sync_text)
        self.header_sync_label.SetLabel(sync_text)

        # Update changes count
        total_changes = len(status.staged_files) + len(status.modified_files) + len(status.untracked_files)
        if total_changes > 0:
            self.changes_label.SetLabel(f"[{total_changes} changes]")
            self.changes_header.SetLabel(f"CHANGES ({total_changes})")
        else:
            self.changes_label.SetLabel("")
            self.changes_header.SetLabel("CHANGES (0)")

        # Update files list
        self.files_list.DeleteAllItems()
        for f in status.staged_files:
            idx = self.files_list.InsertItem(self.files_list.GetItemCount(), "A")
            self.files_list.SetItem(idx, 1, f)
            self.files_list.SetItemTextColour(idx, wx.Colour(34, 197, 94))  # Green
        for f in status.modified_files:
            idx = self.files_list.InsertItem(self.files_list.GetItemCount(), "M")
            self.files_list.SetItem(idx, 1, f)
            self.files_list.SetItemTextColour(idx, wx.Colour(234, 179, 8))  # Yellow
        for f in status.untracked_files:
            idx = self.files_list.InsertItem(self.files_list.GetItemCount(), "?")
            self.files_list.SetItem(idx, 1, f)
            self.files_list.SetItemTextColour(idx, wx.Colour(150, 150, 150))  # Gray

        # Update button states
        self.commit_btn.Enable(total_changes > 0)
        self.push_btn.Enable(status.can_push)
        self.pull_btn.Enable(status.is_repo and status.has_remote)

        # Show/hide section push button based on unpushed commits
        if status.ahead > 0 or status.can_push:
            self.section_push_btn.Show()
            self.section_push_btn.Enable(status.can_push)
            # Update button label with commit count
            if status.ahead > 0:
                self.section_push_btn.SetLabel(f"↑ Push ({status.ahead})")
            else:
                self.section_push_btn.SetLabel("↑ Push")
        else:
            self.section_push_btn.Hide()
        self.section_push_btn.GetParent().Layout()

        # Refresh git graph
        if hasattr(self, 'git_graph'):
            self.git_graph.refresh()

        self.main_panel.Layout()

    def _show_status(self, message: str, status_type: str = STATUS_INFO):
        """Show status message."""
        colors = {
            STATUS_SUCCESS: wx.Colour(34, 197, 94),
            STATUS_WARNING: wx.Colour(234, 179, 8),
            STATUS_ERROR: wx.Colour(239, 68, 68),
            STATUS_INFO: wx.Colour(150, 150, 150),
        }
        self.status_text.SetForegroundColour(colors.get(status_type, colors[STATUS_INFO]))
        self.status_text.SetLabel(message)

    def _get_board_path(self) -> Optional[str]:
        """Get the path to the currently open board file."""
        try:
            board = pcbnew.GetBoard()
            if board:
                return board.GetFileName()
        except Exception:
            pass
        return None

    def _reload_board(self) -> bool:
        """
        Attempt to reload the board using keyboard simulation.

        Simulates Ctrl+W (or Cmd+W on macOS) to close the board,
        then prompts the user and simulates Ctrl+O to open the file dialog.

        Returns True if successful, False to trigger manual prompt fallback.
        """
        board_path = self._get_board_path()
        if not board_path:
            return False

        try:
            sim = wx.UIActionSimulator()

            # Determine modifier key based on platform
            if sys.platform == 'darwin':
                modifier = wx.MOD_CMD
            else:
                modifier = wx.MOD_CONTROL

            # Close current board (Ctrl+W or Cmd+W)
            sim.Char(ord('W'), modifier)

            # Give KiCad time to close
            wx.MilliSleep(500)
            wx.GetApp().Yield()

            # Show dialog to user before opening file dialog
            dlg = wx.MessageDialog(
                self,
                f"Board closed.\n\nClick OK to open the file dialog, then select:\n{board_path}",
                "Reopen Board",
                wx.OK | wx.CANCEL | wx.ICON_INFORMATION
            )

            if dlg.ShowModal() == wx.ID_OK:
                # Simulate Ctrl+O to open file dialog
                sim.Char(ord('O'), modifier)

            dlg.Destroy()
            return True

        except Exception:
            return False

    def _prompt_reload_board(self, context: str):
        """Show dialog prompting user to reload the board."""
        msg = (
            f"{context}\n\n"
            "The board file has changed on disk.\n\n"
            "To see the updated version, please:\n"
            "  1. Close the current board (File → Close)\n"
            "  2. Reopen it (File → Open Recent)\n\n"
            "KiCad will then show the version from the new branch."
        )
        wx.MessageBox(msg, "Board Changed - Reload Required", wx.OK | wx.ICON_INFORMATION)

    def _handle_board_reload_if_needed(self, old_hash: Optional[str], context: str) -> bool:
        """
        Check if board file changed and handle reload.

        Args:
            old_hash: Hash of board file before the operation
            context: Description of the operation (e.g., "Switched to branch 'main'")

        Returns:
            True if board was reloaded or unchanged, False if manual reload needed.
        """
        board_path = self._get_board_path()
        if not board_path or not old_hash:
            return True

        new_hash = self.git.get_file_hash(board_path)
        if not new_hash or old_hash == new_hash:
            return True  # No change

        # Board changed - try to auto-reload
        if self._reload_board():
            self._show_status(f"{context} - board reloaded", STATUS_SUCCESS)
            return True
        else:
            # Fallback: prompt user
            self._prompt_reload_board(context)
            self._show_status(f"{context} - manual reload needed", STATUS_WARNING)
            return False

    def _on_refresh(self, event):
        """Handle refresh button."""
        self._refresh_status()
        self._show_status("Status refreshed", STATUS_SUCCESS)

    def _on_pull(self, event):
        """Handle pull button."""
        self.pull_btn.Disable()

        # Get board file hash before pull
        board_path = self._get_board_path()
        old_hash = self.git.get_file_hash(board_path) if board_path else None

        try:
            result = self.git.pull()
            if result.warning:
                wx.MessageBox(f"Pull completed with warning:\n{result.warning}", "Warning", wx.OK | wx.ICON_WARNING)
                self._show_status("Pull completed (SSL warning)", STATUS_WARNING)
            else:
                self._show_status("Pull successful", STATUS_SUCCESS)

            # Check if board needs reload
            self._handle_board_reload_if_needed(old_hash, "Pull successful")
        except GitError as e:
            error_msg = str(e)

            # Check if it's a conflict error - offer rebase or merge choice
            if 'CONFLICT' in error_msg or 'conflict' in error_msg.lower():
                status = self.git.get_status()
                dlg = SyncStrategyDialog(self, status.ahead, status.behind)
                modal_result = dlg.ShowModal()
                strategy = dlg.strategy
                dlg.Destroy()

                if modal_result == wx.ID_OK and strategy != SyncStrategyDialog.STRATEGY_CANCEL:
                    if self._sync_with_remote(strategy):
                        self._handle_board_reload_if_needed(old_hash, "Pull successful")
                else:
                    self._show_status("Pull cancelled", STATUS_WARNING)
            else:
                wx.MessageBox(f"Pull failed: {e}", "Error", wx.OK | wx.ICON_ERROR)
                self._show_status("Pull failed", STATUS_ERROR)
        finally:
            self.pull_btn.Enable()
            self._refresh_status()

    def _on_push(self, event):
        """Handle push button."""
        self.push_btn.Disable()
        try:
            status = self.git.get_status()

            # Check if we need to sync first
            if status.behind > 0:
                if status.ahead > 0:
                    # Branch has diverged - show sync strategy dialog
                    dlg = SyncStrategyDialog(self, status.ahead, status.behind)
                    result = dlg.ShowModal()
                    strategy = dlg.strategy
                    dlg.Destroy()

                    if result != wx.ID_OK or strategy == SyncStrategyDialog.STRATEGY_CANCEL:
                        self._show_status("Push cancelled", STATUS_WARNING)
                        return

                    if not self._sync_with_remote(strategy):
                        return  # Sync failed or was cancelled
                else:
                    # Just behind - auto-pull first
                    self._show_status("Pulling changes...", STATUS_INFO)
                    try:
                        self.git.pull()
                    except GitError as e:
                        wx.MessageBox(f"Pull failed: {e}", "Error", wx.OK | wx.ICON_ERROR)
                        self._show_status("Pull failed", STATUS_ERROR)
                        return

            # Now push
            result = self.git.push()
            if result.warning:
                wx.MessageBox(f"Push completed with warning:\n{result.warning}", "Warning", wx.OK | wx.ICON_WARNING)
                self._show_status("Push completed (SSL warning)", STATUS_WARNING)
            else:
                self._show_status("Push successful", STATUS_SUCCESS)
        except GitError as e:
            wx.MessageBox(f"Push failed: {e}", "Error", wx.OK | wx.ICON_ERROR)
            self._show_status(f"Push failed", STATUS_ERROR)
        finally:
            self.push_btn.Enable()
            self._refresh_status()

    def _sync_with_remote(self, strategy: str) -> bool:
        """
        Sync local branch with remote using the chosen strategy.

        Args:
            strategy: SyncStrategyDialog.STRATEGY_REBASE or STRATEGY_MERGE

        Returns:
            True if sync succeeded, False otherwise
        """
        if strategy == SyncStrategyDialog.STRATEGY_REBASE:
            self._show_status("Rebasing...", STATUS_INFO)
            result = self.git.pull_rebase()

            if result.has_conflicts:
                # Show rebase conflict dialog
                dlg = RebaseConflictDialog(self, self.git, result.conflicts)
                modal_result = dlg.ShowModal()
                dlg.Destroy()

                if modal_result == wx.ID_CANCEL:
                    # User aborted rebase
                    try:
                        self.git.abort_rebase()
                        self._show_status("Rebase aborted", STATUS_WARNING)
                    except GitError:
                        pass
                    return False

                # User resolved conflicts and clicked Continue Rebase
                try:
                    continue_result = self.git.continue_rebase()
                    if not continue_result.success:
                        wx.MessageBox(
                            f"Failed to continue rebase: {continue_result.message}",
                            "Error", wx.OK | wx.ICON_ERROR
                        )
                        return False
                except GitError as e:
                    wx.MessageBox(f"Failed to continue rebase: {e}", "Error", wx.OK | wx.ICON_ERROR)
                    return False

            if not result.success and not result.has_conflicts:
                wx.MessageBox(f"Rebase failed: {result.message}", "Error", wx.OK | wx.ICON_ERROR)
                self._show_status("Rebase failed", STATUS_ERROR)
                return False

            self._show_status("Rebase successful", STATUS_SUCCESS)
            return True

        else:  # STRATEGY_MERGE
            self._show_status("Merging...", STATUS_INFO)
            try:
                self.git.pull()

                # Check if we're in a conflict state after pull
                if self.git.is_merging():
                    conflicts = self.git.get_conflicting_files()
                    if conflicts:
                        # Show existing merge conflict dialog
                        dlg = MergeConflictDialog(self, self.git, conflicts)
                        modal_result = dlg.ShowModal()
                        dlg.Destroy()

                        if modal_result == wx.ID_CANCEL:
                            # User aborted merge
                            try:
                                self.git.abort_merge()
                                self._show_status("Merge aborted", STATUS_WARNING)
                            except GitError:
                                pass
                            return False

                self._show_status("Merge successful", STATUS_SUCCESS)
                return True

            except GitError as e:
                error_msg = str(e)
                # Check if it's a conflict error
                if 'CONFLICT' in error_msg or 'conflict' in error_msg.lower():
                    conflicts = self.git.get_conflicting_files()
                    if conflicts:
                        dlg = MergeConflictDialog(self, self.git, conflicts)
                        modal_result = dlg.ShowModal()
                        dlg.Destroy()

                        if modal_result == wx.ID_CANCEL:
                            try:
                                self.git.abort_merge()
                                self._show_status("Merge aborted", STATUS_WARNING)
                            except GitError:
                                pass
                            return False

                        self._show_status("Merge successful", STATUS_SUCCESS)
                        return True

                wx.MessageBox(f"Merge failed: {e}", "Error", wx.OK | wx.ICON_ERROR)
                self._show_status("Merge failed", STATUS_ERROR)
                return False

    def _on_commit(self, event):
        """Handle commit button."""
        message = self.commit_input.GetValue().strip()
        if not message:
            message = self.project_config.default_commit_message

        try:
            # Save board first
            board = pcbnew.GetBoard()
            if board and board.GetFileName():
                try:
                    pcbnew.SaveBoard(board.GetFileName(), board)
                except Exception as e:
                    wx.MessageBox(f"Failed to save board: {e}", "Error", wx.OK | wx.ICON_ERROR)
                    return

            self.git.add()  # Stage all changes
            commit_hash = self.git.commit(message)
            self.commit_input.SetValue("")
            self._show_status(f"Saved & committed: {commit_hash[:8]}", STATUS_SUCCESS)
            self._refresh_status()
        except GitError as e:
            wx.MessageBox(f"Commit failed: {e}", "Error", wx.OK | wx.ICON_ERROR)
            self._show_status("Commit failed", STATUS_ERROR)

    def _on_settings_menu(self, event):
        """Show settings popup menu."""
        menu = wx.Menu()

        # Auto-save toggle
        auto_save_id = wx.NewId()
        auto_save_item = menu.AppendCheckItem(auto_save_id, "Auto-save on open")
        auto_save_item.Check(self.project_config.auto_commit_on_focus)
        self.Bind(wx.EVT_MENU, self._on_toggle_auto_save, id=auto_save_id)

        # Auto-pull toggle
        auto_pull_id = wx.NewId()
        auto_pull_item = menu.AppendCheckItem(auto_pull_id, "Auto-pull on open")
        auto_pull_item.Check(self.project_config.auto_pull_on_open)
        self.Bind(wx.EVT_MENU, self._on_toggle_auto_pull, id=auto_pull_id)

        menu.AppendSeparator()

        # Initialize repo (disabled if repo already exists)
        init_id = wx.NewId()
        init_item = menu.Append(init_id, "Initialize Repository")
        self.Bind(wx.EVT_MENU, self._on_init_repo, id=init_id)
        status = self.git.get_status()
        init_item.Enable(not status.is_repo)

        # Manage remotes
        remotes_id = wx.NewId()
        remotes_item = menu.Append(remotes_id, "Manage Remotes...")
        self.Bind(wx.EVT_MENU, self._on_manage_remotes, id=remotes_id)

        # Branch management
        menu.AppendSeparator()
        new_branch_id = wx.NewId()
        menu.Append(new_branch_id, "New Branch...")
        self.Bind(wx.EVT_MENU, self._on_new_branch, id=new_branch_id)

        switch_branch_id = wx.NewId()
        menu.Append(switch_branch_id, "Switch Branch...")
        self.Bind(wx.EVT_MENU, self._on_switch_branch, id=switch_branch_id)

        merge_branch_id = wx.NewId()
        menu.Append(merge_branch_id, "Merge Branch...")
        self.Bind(wx.EVT_MENU, self._on_merge_branch, id=merge_branch_id)

        menu.AppendSeparator()

        # Login/Logout based on auth state
        if self.config.is_authenticated():
            username = self.config.gitea_username or "User"
            logout_id = wx.NewId()
            menu.Append(logout_id, f"Logout as {username}")
            self.Bind(wx.EVT_MENU, self._on_logout, id=logout_id)
        else:
            login_id = wx.NewId()
            menu.Append(login_id, "Login...")
            self.Bind(wx.EVT_MENU, self._on_login, id=login_id)

        menu.AppendSeparator()

        # Full settings dialog
        settings_id = wx.NewId()
        menu.Append(settings_id, "Settings...")
        self.Bind(wx.EVT_MENU, self._on_open_settings, id=settings_id)

        self.PopupMenu(menu)
        menu.Destroy()

    def _on_toggle_auto_save(self, event):
        """Toggle auto-save setting."""
        self.project_config.auto_commit_on_focus = not self.project_config.auto_commit_on_focus
        status = "enabled" if self.project_config.auto_commit_on_focus else "disabled"
        self._show_status(f"Auto-save {status}", STATUS_INFO)

    def _on_toggle_auto_pull(self, event):
        """Toggle auto-pull setting."""
        self.project_config.auto_pull_on_open = not self.project_config.auto_pull_on_open
        status = "enabled" if self.project_config.auto_pull_on_open else "disabled"
        self._show_status(f"Auto-pull {status}", STATUS_INFO)

    def _on_open_settings(self, event):
        """Open full settings dialog."""
        dlg = wx.Dialog(self, title="Settings", size=(450, 400))
        panel = SettingsPanel(dlg, self.config, self.project_config,
                              on_settings_changed=self._refresh_status)
        sizer = wx.BoxSizer(wx.VERTICAL)
        sizer.Add(panel, 1, wx.EXPAND)
        dlg.SetSizer(sizer)
        dlg.ShowModal()
        dlg.Destroy()

    def _on_init_repo(self, event):
        """Initialize a new repository."""
        if wx.MessageBox(
            "Initialize a new Git repository in this project?",
            "Initialize Repository",
            wx.YES_NO | wx.ICON_QUESTION
        ) == wx.YES:
            try:
                self.git.init()
                self._show_status("Repository initialized", STATUS_SUCCESS)
                self._refresh_status()
            except GitError as e:
                wx.MessageBox(f"Failed to initialize: {e}", "Error", wx.OK | wx.ICON_ERROR)

    def _on_manage_remotes(self, event):
        """Open remotes management dialog."""
        dlg = RemoteManagementDialog(self, self.git, self.config, self.project_config)
        dlg.ShowModal()
        dlg.Destroy()
        self._refresh_status()

    def _on_new_branch(self, event):
        """Create new branch."""
        dlg = BranchDialog(self, self.git, mode="create")
        if dlg.ShowModal() == wx.ID_OK:
            branch_name = getattr(dlg, 'result_branch', '')
            self._show_status(f"Branch '{branch_name}' created", STATUS_SUCCESS)
            self._refresh_status()
        dlg.Destroy()

    def _on_switch_branch(self, event):
        """Switch branch."""
        # Get board file hash before switch
        board_path = self._get_board_path()
        old_hash = self.git.get_file_hash(board_path) if board_path else None

        dlg = BranchDialog(self, self.git, mode="switch")
        if dlg.ShowModal() == wx.ID_OK:
            branch_name = getattr(dlg, 'result_branch', '')
            # Check if board needs reload
            if not self._handle_board_reload_if_needed(old_hash, f"Switched to '{branch_name}'"):
                pass  # Status already shown by handler
            else:
                self._show_status(f"Switched to '{branch_name}'", STATUS_SUCCESS)
            self._refresh_status()
        dlg.Destroy()

    def _on_merge_branch(self, event):
        """Merge another branch into current branch."""
        # Get board file hash before merge
        board_path = self._get_board_path()
        old_hash = self.git.get_file_hash(board_path) if board_path else None

        # Show branch selection dialog
        dlg = BranchDialog(self, self.git, mode="merge")
        if dlg.ShowModal() == wx.ID_OK:
            branch_name = getattr(dlg, 'result_branch', '')
            try:
                result = self.git.merge(branch_name)

                if result.success:
                    # Check if board needs reload
                    self._handle_board_reload_if_needed(old_hash, f"Merged '{branch_name}'")
                    self._show_status(result.message, STATUS_SUCCESS)
                else:
                    # Conflicts detected
                    self._handle_merge_conflicts(result.conflicts, branch_name)

            except GitError as e:
                wx.MessageBox(f"Merge failed: {e}", "Error", wx.OK | wx.ICON_ERROR)
                self._show_status("Merge failed", STATUS_ERROR)

            self._refresh_status()
        dlg.Destroy()

    def _handle_merge_conflicts(self, conflicts: list, branch_name: str):
        """Handle merge conflicts by showing resolution dialog."""
        dlg = MergeConflictDialog(self, self.git, conflicts, branch_name)
        result = dlg.ShowModal()

        if result == wx.ID_OK:
            # Conflicts resolved, merge completed
            self._show_status("Merge completed", STATUS_SUCCESS)
        elif result == wx.ID_CANCEL:
            # User aborted merge
            try:
                self.git.abort_merge()
                self._show_status("Merge aborted", STATUS_WARNING)
            except GitError:
                pass

        dlg.Destroy()

    def _on_login(self, event):
        """Start OAuth login flow."""
        if not self.config.base_url:
            dlg = SetupDialog(self, self.config)
            if dlg.ShowModal() != wx.ID_SAVE:
                dlg.Destroy()
                return
            dlg.Destroy()

        self._show_status("Connecting...", STATUS_INFO)
        self._oauth_handler = OAuthHandler(self.config.base_url)

        def on_complete(tokens: OAuthTokens):
            evt = AuthCompleteEvent(tokens=tokens)
            wx.PostEvent(self, evt)

        def on_error(error: str):
            evt = AuthErrorEvent(error=error)
            wx.PostEvent(self, evt)

        def start_flow():
            try:
                self._oauth_handler.discover()
                if not self._oauth_handler.start_auth_flow(on_complete, on_error):
                    on_error("Failed to start authentication flow")
            except OAuthError as e:
                on_error(f"Discovery failed: {e}")
            except Exception as e:
                on_error(f"Connection error: {e}")

        thread = threading.Thread(target=start_flow)
        thread.daemon = True
        thread.start()

    def _on_auth_complete(self, event):
        """Handle successful authentication."""
        tokens = event.tokens
        self.config.access_token = tokens.access_token
        if tokens.refresh_token:
            self.config.refresh_token = tokens.refresh_token
        if tokens.expires_in:
            import time
            self.config.token_expires_at = int(time.time()) + tokens.expires_in

        try:
            user_info = self._oauth_handler.get_user_info(tokens.access_token)
            self.config.gitea_access_token = user_info.gitea_access_token
            self.config.gitea_username = user_info.gitea_username
            self.config.gitea_org_name = user_info.gitea_org_name
            self.config.gitea_url = self._oauth_handler.get_gitea_url()
            self.config.user_info = user_info.raw_data
        except OAuthError as e:
            wx.MessageBox(f"Failed to get user info: {e}", "Warning", wx.OK | wx.ICON_WARNING)

        self._show_status("Authenticated successfully", STATUS_SUCCESS)

        ssl_warning = get_oauth_ssl_warning()
        if ssl_warning:
            wx.MessageBox(f"Authenticated with warning:\n{ssl_warning}", "Warning", wx.OK | wx.ICON_WARNING)
            clear_oauth_ssl_warning()

    def _on_auth_error(self, event):
        """Handle authentication error."""
        wx.MessageBox(f"Authentication failed: {event.error}", "Error", wx.OK | wx.ICON_ERROR)
        self._show_status("Authentication failed", STATUS_ERROR)

    def _on_logout(self, event):
        """Handle logout."""
        if wx.MessageBox("Are you sure you want to logout?", "Confirm", wx.YES_NO | wx.ICON_QUESTION) == wx.YES:
            self.config.clear_auth()
            self._show_status("Logged out", STATUS_INFO)


class GitNotInstalledDialog(wx.Dialog):
    """Dialog shown when Git is not installed on the system."""

    def __init__(self, parent):
        super().__init__(parent, title="Git Not Found", size=(500, 350))

        self.SetMinSize((500, 350))

        panel = wx.Panel(self)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Warning icon and title
        title_sizer = wx.BoxSizer(wx.HORIZONTAL)
        warning_text = wx.StaticText(panel, label="Git is not installed")
        warning_text.SetFont(wx.Font(14, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        title_sizer.Add(warning_text, 0, wx.ALL, 10)
        main_sizer.Add(title_sizer, 0, wx.ALIGN_CENTER)

        # Description
        desc = wx.StaticText(panel, label=(
            "Git is required for version control functionality.\n"
            "Please install Git to use this plugin."
        ))
        main_sizer.Add(desc, 0, wx.ALL | wx.ALIGN_CENTER, 10)

        # Installation instructions
        instructions_box = wx.StaticBox(panel, label="Installation Instructions")
        instructions_sizer = wx.StaticBoxSizer(instructions_box, wx.VERTICAL)

        instructions = GitUtils.get_install_instructions()
        instructions_text = wx.TextCtrl(
            panel, value=instructions,
            style=wx.TE_MULTILINE | wx.TE_READONLY | wx.TE_DONTWRAP
        )
        instructions_text.SetMinSize((-1, 100))
        instructions_sizer.Add(instructions_text, 1, wx.EXPAND | wx.ALL, 5)

        main_sizer.Add(instructions_sizer, 1, wx.EXPAND | wx.ALL, 10)

        # Download button
        download_btn = wx.Button(panel, label="Open Git Download Page")
        download_btn.Bind(wx.EVT_BUTTON, self._on_download)
        main_sizer.Add(download_btn, 0, wx.ALIGN_CENTER | wx.ALL, 10)

        # Close button
        close_btn = wx.Button(panel, wx.ID_CLOSE, "Close")
        close_btn.Bind(wx.EVT_BUTTON, lambda e: self.EndModal(wx.ID_CLOSE))
        main_sizer.Add(close_btn, 0, wx.ALIGN_CENTER | wx.BOTTOM, 10)

        panel.SetSizer(main_sizer)
        self.Centre()

    def _on_download(self, event):
        """Open Git download page in browser."""
        import webbrowser
        webbrowser.open(GitUtils.GIT_DOWNLOAD_URL)


class SetupDialog(wx.Dialog):
    """Dialog for initial plugin setup - configuring the Paplix service URL."""

    def __init__(self, parent, config: Config):
        super().__init__(parent, title="Paplix Setup", size=(450, 250))

        self.config = config
        self.SetMinSize((450, 250))

        panel = wx.Panel(self)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Title
        title = wx.StaticText(panel, label="Configure Paplix Service")
        title.SetFont(wx.Font(12, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        main_sizer.Add(title, 0, wx.ALL | wx.ALIGN_CENTER, 15)

        # Description
        desc = wx.StaticText(panel, label=(
            "Enter the URL of your Paplix version control service.\n"
            "This is where your repositories and authentication will be hosted."
        ))
        main_sizer.Add(desc, 0, wx.ALL | wx.ALIGN_CENTER, 10)

        # URL input
        url_sizer = wx.BoxSizer(wx.HORIZONTAL)
        url_label = wx.StaticText(panel, label="Service URL:")
        self.url_input = wx.TextCtrl(panel, size=(300, -1))
        if self.config.base_url:
            self.url_input.SetValue(self.config.base_url)
        else:
            self.url_input.SetHint("https://your-paplix-instance.com")

        url_sizer.Add(url_label, 0, wx.ALIGN_CENTER_VERTICAL | wx.RIGHT, 10)
        url_sizer.Add(self.url_input, 1, wx.EXPAND)
        main_sizer.Add(url_sizer, 0, wx.EXPAND | wx.ALL, 15)

        # Buttons
        btn_sizer = wx.BoxSizer(wx.HORIZONTAL)
        save_btn = wx.Button(panel, wx.ID_SAVE, "Save")
        cancel_btn = wx.Button(panel, wx.ID_CANCEL, "Cancel")

        save_btn.Bind(wx.EVT_BUTTON, self._on_save)
        cancel_btn.Bind(wx.EVT_BUTTON, lambda e: self.EndModal(wx.ID_CANCEL))

        btn_sizer.Add(cancel_btn, 0, wx.RIGHT, 10)
        btn_sizer.Add(save_btn, 0)
        main_sizer.Add(btn_sizer, 0, wx.ALIGN_CENTER | wx.ALL, 15)

        panel.SetSizer(main_sizer)
        self.Centre()

    def _on_save(self, event):
        """Save the URL configuration."""
        url = self.url_input.GetValue().strip()

        if not url:
            wx.MessageBox("Please enter a URL.", "Validation Error", wx.OK | wx.ICON_WARNING)
            return

        # Basic URL validation
        if not url.startswith(('http://', 'https://')):
            wx.MessageBox(
                "Please enter a valid URL starting with http:// or https://",
                "Validation Error",
                wx.OK | wx.ICON_WARNING
            )
            return

        self.config.base_url = url
        self.EndModal(wx.ID_SAVE)


class AuthPanel(wx.Panel):
    """Panel for authentication status and login/logout."""

    def __init__(self, parent, config: Config, on_auth_changed: Optional[Callable] = None):
        super().__init__(parent)

        self.config = config
        self.on_auth_changed = on_auth_changed
        self._oauth_handler: Optional[OAuthHandler] = None

        self._create_ui()
        self._update_auth_status()

        # Bind custom events
        self.Bind(EVT_AUTH_COMPLETE, self._on_auth_complete)
        self.Bind(EVT_AUTH_ERROR, self._on_auth_error)

    def _create_ui(self):
        """Create the authentication panel UI."""
        sizer = wx.StaticBoxSizer(wx.VERTICAL, self, "Authentication")

        # Status display
        self.status_text = wx.StaticText(self, label="Not authenticated")
        sizer.Add(self.status_text, 0, wx.ALL, 10)

        # User info (when authenticated)
        self.user_info_text = wx.StaticText(self, label="")
        sizer.Add(self.user_info_text, 0, wx.LEFT | wx.RIGHT | wx.BOTTOM, 10)

        # Buttons
        btn_sizer = wx.BoxSizer(wx.HORIZONTAL)
        self.login_btn = wx.Button(self, label="Login with Paplix")
        self.logout_btn = wx.Button(self, label="Logout")
        self.settings_btn = wx.Button(self, label="Settings")

        self.login_btn.Bind(wx.EVT_BUTTON, self._on_login)
        self.logout_btn.Bind(wx.EVT_BUTTON, self._on_logout)
        self.settings_btn.Bind(wx.EVT_BUTTON, self._on_settings)

        btn_sizer.Add(self.login_btn, 0, wx.RIGHT, 5)
        btn_sizer.Add(self.logout_btn, 0, wx.RIGHT, 5)
        btn_sizer.Add(self.settings_btn, 0)

        sizer.Add(btn_sizer, 0, wx.ALL, 10)

        self.SetSizer(sizer)

    def _update_auth_status(self):
        """Update the authentication status display."""
        if self.config.is_authenticated():
            self.status_text.SetLabel("Authenticated")
            self.status_text.SetForegroundColour(wx.Colour(34, 197, 94))  # Green

            # Show Gitea username and org
            username = self.config.gitea_username
            org_name = self.config.gitea_org_name
            if username:
                user_text = f"User: {username}"
                if org_name:
                    user_text += f" (Org: {org_name})"
                self.user_info_text.SetLabel(user_text)
            else:
                self.user_info_text.SetLabel("")

            self.login_btn.Hide()
            self.logout_btn.Show()
        else:
            self.status_text.SetLabel("Not authenticated")
            self.status_text.SetForegroundColour(wx.Colour(107, 114, 128))  # Gray

            self.user_info_text.SetLabel("")
            self.login_btn.Show()
            self.logout_btn.Hide()

        self.Layout()
        self.GetParent().Layout()

    def _on_login(self, event):
        """Start OAuth login flow."""
        if not self.config.base_url:
            wx.MessageBox(
                "Please configure the service URL first.",
                "Configuration Required",
                wx.OK | wx.ICON_WARNING
            )
            return

        self.login_btn.Disable()
        self.login_btn.SetLabel("Connecting...")

        self._oauth_handler = OAuthHandler(self.config.base_url)

        def on_complete(tokens: OAuthTokens):
            evt = AuthCompleteEvent(tokens=tokens)
            wx.PostEvent(self, evt)

        def on_error(error: str):
            evt = AuthErrorEvent(error=error)
            wx.PostEvent(self, evt)

        # Run discovery and auth flow in background thread to not block UI
        def start_flow():
            try:
                # First discover endpoints (this may fail if server unreachable)
                self._oauth_handler.discover()
                # Then start the auth flow
                if not self._oauth_handler.start_auth_flow(on_complete, on_error):
                    on_error("Failed to start authentication flow")
            except OAuthError as e:
                on_error(f"Discovery failed: {e}")
            except Exception as e:
                on_error(f"Connection error: {e}")

        thread = threading.Thread(target=start_flow)
        thread.daemon = True
        thread.start()

    def _on_auth_complete(self, event):
        """Handle successful authentication."""
        tokens = event.tokens
        self.config.access_token = tokens.access_token
        if tokens.refresh_token:
            self.config.refresh_token = tokens.refresh_token
        if tokens.expires_in:
            import time
            self.config.token_expires_at = int(time.time()) + tokens.expires_in

        # Fetch user info with Gitea credentials
        try:
            user_info = self._oauth_handler.get_user_info(tokens.access_token)
            # Store Gitea credentials
            self.config.gitea_access_token = user_info.gitea_access_token
            self.config.gitea_username = user_info.gitea_username
            self.config.gitea_org_name = user_info.gitea_org_name
            self.config.gitea_url = self._oauth_handler.get_gitea_url()
            self.config.user_info = user_info.raw_data
        except OAuthError as e:
            wx.MessageBox(
                f"Authentication succeeded but failed to get user info: {e}",
                "Warning",
                wx.OK | wx.ICON_WARNING
            )

        self.login_btn.Enable()
        self.login_btn.SetLabel("Login with Paplix")
        self._update_auth_status()

        # Check for SSL warnings - only show dialog for warnings
        ssl_warning = get_oauth_ssl_warning()
        if ssl_warning:
            wx.MessageBox(
                f"Successfully authenticated with Paplix!\n\nWarning: {ssl_warning}",
                "Authentication Success (with warning)",
                wx.OK | wx.ICON_WARNING
            )
            clear_oauth_ssl_warning()
        # No success dialog for normal auth - status bar will show it

        if self.on_auth_changed:
            self.on_auth_changed(auth_success=True, ssl_warning=ssl_warning)

    def _on_auth_error(self, event):
        """Handle authentication error."""
        self.login_btn.Enable()
        self.login_btn.SetLabel("Login with Paplix")

        wx.MessageBox(
            f"Authentication failed: {event.error}",
            "Authentication Error",
            wx.OK | wx.ICON_ERROR
        )

    def _on_logout(self, event):
        """Handle logout."""
        if wx.MessageBox(
            "Are you sure you want to logout?",
            "Confirm Logout",
            wx.YES_NO | wx.ICON_QUESTION
        ) == wx.YES:
            self.config.clear_auth()
            self._update_auth_status()
            if self.on_auth_changed:
                self.on_auth_changed(logged_out=True)

    def _on_settings(self, event):
        """Open settings dialog."""
        dlg = SetupDialog(self, self.config)
        if dlg.ShowModal() == wx.ID_SAVE:
            # URL changed, clear auth as it's likely invalid now
            self.config.clear_auth()
            self._update_auth_status()
            if self.on_auth_changed:
                self.on_auth_changed(url_changed=True)
        dlg.Destroy()


class GitStatusPanel(wx.Panel):
    """Panel showing current Git repository status."""

    def __init__(self, parent, git: GitUtils):
        super().__init__(parent)

        self.git = git
        self._create_ui()

    def _create_ui(self):
        """Create the status panel UI."""
        sizer = wx.StaticBoxSizer(wx.VERTICAL, self, "Repository Status")

        # Status info
        grid = wx.FlexGridSizer(rows=5, cols=2, hgap=10, vgap=5)
        grid.AddGrowableCol(1)

        self.repo_label = wx.StaticText(self, label="Repository:")
        self.repo_value = wx.StaticText(self, label="Not initialized")

        self.branch_label = wx.StaticText(self, label="Branch:")
        self.branch_value = wx.StaticText(self, label="-")

        self.changes_label = wx.StaticText(self, label="Changes:")
        self.changes_value = wx.StaticText(self, label="-")

        self.remote_label = wx.StaticText(self, label="Remote:")
        self.remote_value = wx.StaticText(self, label="-")

        self.sync_label = wx.StaticText(self, label="Sync Status:")
        self.sync_value = wx.StaticText(self, label="-")

        grid.AddMany([
            (self.repo_label, 0), (self.repo_value, 1, wx.EXPAND),
            (self.branch_label, 0), (self.branch_value, 1, wx.EXPAND),
            (self.changes_label, 0), (self.changes_value, 1, wx.EXPAND),
            (self.remote_label, 0), (self.remote_value, 1, wx.EXPAND),
            (self.sync_label, 0), (self.sync_value, 1, wx.EXPAND),
        ])

        sizer.Add(grid, 1, wx.EXPAND | wx.ALL, 10)

        # Refresh button
        refresh_btn = wx.Button(self, label="Refresh")
        refresh_btn.Bind(wx.EVT_BUTTON, lambda e: self.refresh())
        sizer.Add(refresh_btn, 0, wx.ALIGN_RIGHT | wx.ALL, 10)

        self.SetSizer(sizer)

    def refresh(self) -> GitStatus:
        """Refresh and display current Git status."""
        status = self.git.get_status()

        if status.is_repo:
            self.repo_value.SetLabel("Initialized")
            self.repo_value.SetForegroundColour(wx.Colour(34, 197, 94))  # Green

            self.branch_value.SetLabel(status.branch or "detached HEAD")

            # Changes summary
            total_changes = len(status.staged_files) + len(status.modified_files) + len(status.untracked_files)
            if total_changes > 0:
                changes_text = f"{total_changes} file(s) changed"
                self.changes_value.SetLabel(changes_text)
                self.changes_value.SetForegroundColour(wx.Colour(234, 179, 8))  # Yellow
            else:
                self.changes_value.SetLabel("No changes")
                self.changes_value.SetForegroundColour(wx.Colour(34, 197, 94))  # Green

            # Remote
            if status.has_remote:
                self.remote_value.SetLabel(status.remote_url[:50] + "..." if len(status.remote_url) > 50 else status.remote_url)

                # Sync status
                if status.ahead > 0 and status.behind > 0:
                    sync_text = f"{status.ahead} ahead, {status.behind} behind"
                    self.sync_value.SetForegroundColour(wx.Colour(234, 179, 8))  # Yellow
                elif status.ahead > 0:
                    sync_text = f"{status.ahead} commit(s) to push"
                    self.sync_value.SetForegroundColour(wx.Colour(234, 179, 8))
                elif status.behind > 0:
                    sync_text = f"{status.behind} commit(s) to pull"
                    self.sync_value.SetForegroundColour(wx.Colour(234, 179, 8))
                else:
                    sync_text = "Up to date"
                    self.sync_value.SetForegroundColour(wx.Colour(34, 197, 94))  # Green
                self.sync_value.SetLabel(sync_text)
            else:
                self.remote_value.SetLabel("Not configured")
                self.sync_value.SetLabel("-")
        else:
            self.repo_value.SetLabel("Not initialized")
            self.repo_value.SetForegroundColour(wx.Colour(107, 114, 128))  # Gray
            self.branch_value.SetLabel("-")
            self.changes_value.SetLabel("-")
            self.remote_value.SetLabel("-")
            self.sync_value.SetLabel("-")

        self.Layout()
        return status


class CommitDialog(wx.Dialog):
    """Dialog for creating a Git commit."""

    def __init__(self, parent, git: GitUtils, status: GitStatus):
        super().__init__(parent, title="Create Commit", size=(500, 450))

        self.git = git
        self.status = status
        self.SetMinSize((500, 450))

        panel = wx.Panel(self)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Files to commit
        files_box = wx.StaticBox(panel, label="Files to Commit")
        files_sizer = wx.StaticBoxSizer(files_box, wx.VERTICAL)

        self.files_list = wx.CheckListBox(panel, size=(-1, 150))
        self._populate_files()
        files_sizer.Add(self.files_list, 1, wx.EXPAND | wx.ALL, 5)

        # Select all / none buttons
        select_sizer = wx.BoxSizer(wx.HORIZONTAL)
        select_all_btn = wx.Button(panel, label="Select All")
        select_none_btn = wx.Button(panel, label="Select None")
        select_all_btn.Bind(wx.EVT_BUTTON, self._on_select_all)
        select_none_btn.Bind(wx.EVT_BUTTON, self._on_select_none)
        select_sizer.Add(select_all_btn, 0, wx.RIGHT, 5)
        select_sizer.Add(select_none_btn, 0)
        files_sizer.Add(select_sizer, 0, wx.ALL, 5)

        main_sizer.Add(files_sizer, 1, wx.EXPAND | wx.ALL, 10)

        # Commit message
        msg_box = wx.StaticBox(panel, label="Commit Message")
        msg_sizer = wx.StaticBoxSizer(msg_box, wx.VERTICAL)

        self.message_input = wx.TextCtrl(panel, style=wx.TE_MULTILINE, size=(-1, 100))
        self.message_input.SetHint("Enter your commit message here...")
        msg_sizer.Add(self.message_input, 1, wx.EXPAND | wx.ALL, 5)

        main_sizer.Add(msg_sizer, 0, wx.EXPAND | wx.LEFT | wx.RIGHT | wx.BOTTOM, 10)

        # Buttons
        btn_sizer = wx.BoxSizer(wx.HORIZONTAL)
        commit_btn = wx.Button(panel, wx.ID_OK, "Commit")
        cancel_btn = wx.Button(panel, wx.ID_CANCEL, "Cancel")

        commit_btn.Bind(wx.EVT_BUTTON, self._on_commit)

        btn_sizer.Add(cancel_btn, 0, wx.RIGHT, 10)
        btn_sizer.Add(commit_btn, 0)

        main_sizer.Add(btn_sizer, 0, wx.ALIGN_CENTER | wx.ALL, 10)

        panel.SetSizer(main_sizer)
        self.Centre()

    def _populate_files(self):
        """Populate the file list with changed files."""
        all_files = []

        for f in self.status.staged_files:
            all_files.append(f"[staged] {f}")
        for f in self.status.modified_files:
            all_files.append(f"[modified] {f}")
        for f in self.status.untracked_files:
            all_files.append(f"[new] {f}")

        self.files_list.Set(all_files)

        # Select all by default
        for i in range(self.files_list.GetCount()):
            self.files_list.Check(i, True)

    def _on_select_all(self, event):
        """Select all files."""
        for i in range(self.files_list.GetCount()):
            self.files_list.Check(i, True)

    def _on_select_none(self, event):
        """Deselect all files."""
        for i in range(self.files_list.GetCount()):
            self.files_list.Check(i, False)

    def _on_commit(self, event):
        """Create the commit."""
        message = self.message_input.GetValue().strip()
        if not message:
            wx.MessageBox("Please enter a commit message.", "Validation Error", wx.OK | wx.ICON_WARNING)
            return

        # Get selected files
        selected_files = []
        for i in range(self.files_list.GetCount()):
            if self.files_list.IsChecked(i):
                # Extract filename from "[status] filename" format
                item = self.files_list.GetString(i)
                filename = item.split('] ', 1)[1] if '] ' in item else item
                selected_files.append(filename)

        if not selected_files:
            wx.MessageBox("Please select at least one file to commit.", "Validation Error", wx.OK | wx.ICON_WARNING)
            return

        try:
            # Stage selected files
            self.git.add(selected_files)
            # Create commit
            self.commit_hash = self.git.commit(message)
            self.EndModal(wx.ID_OK)
        except GitError as e:
            wx.MessageBox(f"Commit failed: {e}", "Error", wx.OK | wx.ICON_ERROR)


class MergeConflictDialog(wx.Dialog):
    """Dialog for resolving merge conflicts."""

    def __init__(self, parent, git: GitUtils, conflicts: list, branch_name: str):
        """
        Initialize merge conflict dialog.

        Args:
            parent: Parent window
            git: GitUtils instance
            conflicts: List of conflicting file paths
            branch_name: Name of the branch being merged
        """
        super().__init__(parent, title="Merge Conflicts", size=(500, 400))

        self.git = git
        self.conflicts = conflicts[:]  # Copy the list
        self.branch_name = branch_name
        self.resolved_files = set()
        self.SetMinSize((500, 400))

        panel = wx.Panel(self)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Header
        header = wx.StaticText(panel, label=f"Merge conflicts detected ({len(conflicts)} file(s))")
        header.SetFont(wx.Font(12, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        main_sizer.Add(header, 0, wx.ALL, 15)

        # Info text
        info = wx.StaticText(panel, label=(
            f"The following files have conflicts when merging '{branch_name}'.\n"
            "Choose which version to keep for each file."
        ))
        info.SetForegroundColour(wx.Colour(107, 114, 128))
        main_sizer.Add(info, 0, wx.LEFT | wx.RIGHT | wx.BOTTOM, 15)

        # Conflicts list with resolution buttons
        self.conflict_panel = wx.ScrolledWindow(panel)
        self.conflict_panel.SetScrollRate(0, 20)
        conflict_sizer = wx.BoxSizer(wx.VERTICAL)

        self.file_items = {}
        for filepath in conflicts:
            item_sizer = self._create_conflict_item(self.conflict_panel, filepath)
            conflict_sizer.Add(item_sizer, 0, wx.EXPAND | wx.BOTTOM, 10)

        self.conflict_panel.SetSizer(conflict_sizer)
        main_sizer.Add(self.conflict_panel, 1, wx.EXPAND | wx.LEFT | wx.RIGHT, 15)

        # Status text
        self.status_text = wx.StaticText(panel, label="")
        self.status_text.SetForegroundColour(wx.Colour(34, 197, 94))
        main_sizer.Add(self.status_text, 0, wx.ALL, 15)

        # Buttons
        btn_sizer = wx.BoxSizer(wx.HORIZONTAL)
        self.complete_btn = wx.Button(panel, wx.ID_OK, "Complete Merge")
        self.complete_btn.Enable(False)
        abort_btn = wx.Button(panel, wx.ID_CANCEL, "Abort Merge")

        btn_sizer.Add(abort_btn, 0, wx.RIGHT, 10)
        btn_sizer.Add(self.complete_btn, 0)

        main_sizer.Add(btn_sizer, 0, wx.ALIGN_CENTER | wx.BOTTOM, 15)

        panel.SetSizer(main_sizer)
        self.Centre()

        self._update_status()

    def _create_conflict_item(self, parent, filepath: str) -> wx.BoxSizer:
        """Create a conflict item with resolution buttons."""
        item_sizer = wx.BoxSizer(wx.HORIZONTAL)

        # Filename
        filename = os.path.basename(filepath)
        file_label = wx.StaticText(parent, label=filename)
        file_label.SetToolTip(filepath)
        item_sizer.Add(file_label, 1, wx.ALIGN_CENTER_VERTICAL)

        # Resolution buttons
        ours_btn = wx.Button(parent, label="Keep Ours", size=(90, -1))
        theirs_btn = wx.Button(parent, label="Keep Theirs", size=(90, -1))

        ours_btn.Bind(wx.EVT_BUTTON, lambda e, f=filepath: self._resolve_conflict(f, 'ours'))
        theirs_btn.Bind(wx.EVT_BUTTON, lambda e, f=filepath: self._resolve_conflict(f, 'theirs'))

        item_sizer.Add(ours_btn, 0, wx.RIGHT, 5)
        item_sizer.Add(theirs_btn, 0)

        # Store references for updating
        self.file_items[filepath] = {
            'label': file_label,
            'ours_btn': ours_btn,
            'theirs_btn': theirs_btn,
            'resolved': False,
            'choice': None
        }

        return item_sizer

    def _resolve_conflict(self, filepath: str, choice: str):
        """Resolve a single conflict."""
        try:
            self.git.resolve_conflict(filepath, choice)
            self.resolved_files.add(filepath)

            # Update UI for this item
            item = self.file_items[filepath]
            item['resolved'] = True
            item['choice'] = choice
            item['ours_btn'].Enable(False)
            item['theirs_btn'].Enable(False)
            item['label'].SetForegroundColour(wx.Colour(34, 197, 94))
            choice_text = "ours" if choice == 'ours' else "theirs"
            item['label'].SetLabel(f"✓ {os.path.basename(filepath)} (kept {choice_text})")

            self._update_status()
            self.conflict_panel.Refresh()

        except GitError as e:
            wx.MessageBox(f"Failed to resolve conflict: {e}", "Error", wx.OK | wx.ICON_ERROR)

    def _update_status(self):
        """Update status text and button state."""
        resolved_count = len(self.resolved_files)
        total_count = len(self.conflicts)

        if resolved_count == total_count:
            self.status_text.SetLabel(f"All {total_count} conflict(s) resolved. Ready to complete merge.")
            self.complete_btn.Enable(True)
            # Auto-complete the merge commit
            try:
                self.git.complete_merge()
            except GitError:
                pass  # Will be committed when dialog closes
        else:
            remaining = total_count - resolved_count
            self.status_text.SetLabel(f"{remaining} conflict(s) remaining to resolve.")
            self.status_text.SetForegroundColour(wx.Colour(234, 179, 8))
            self.complete_btn.Enable(False)


class SyncStrategyDialog(wx.Dialog):
    """Dialog for choosing sync strategy when branch has diverged."""

    STRATEGY_REBASE = "rebase"
    STRATEGY_MERGE = "merge"
    STRATEGY_CANCEL = "cancel"

    def __init__(self, parent, ahead_count: int, behind_count: int):
        """
        Initialize sync strategy dialog.

        Args:
            parent: Parent window
            ahead_count: Number of commits ahead of remote
            behind_count: Number of commits behind remote
        """
        super().__init__(parent, title="Sync Required", size=(450, 320))

        self.strategy = self.STRATEGY_CANCEL
        self.SetMinSize((450, 320))

        panel = wx.Panel(self)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Header
        header = wx.StaticText(panel, label="Your branch has diverged from remote")
        header.SetFont(wx.Font(12, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        main_sizer.Add(header, 0, wx.ALL, 15)

        # Info text
        info = wx.StaticText(panel, label=(
            f"Your branch is {ahead_count} commit(s) ahead and {behind_count} commit(s) behind.\n"
            "You need to sync before pushing. Choose a strategy:"
        ))
        info.SetForegroundColour(wx.Colour(107, 114, 128))
        main_sizer.Add(info, 0, wx.LEFT | wx.RIGHT | wx.BOTTOM, 15)

        # Rebase option
        rebase_btn = wx.Button(panel, label="Rebase (Recommended)", size=(-1, 35))
        rebase_btn.SetFont(wx.Font(10, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        rebase_desc = wx.StaticText(panel, label=(
            "Replay your commits on top of remote changes.\n"
            "Creates a clean, linear history."
        ))
        rebase_desc.SetForegroundColour(wx.Colour(107, 114, 128))
        rebase_btn.Bind(wx.EVT_BUTTON, self._on_rebase)
        main_sizer.Add(rebase_btn, 0, wx.EXPAND | wx.LEFT | wx.RIGHT, 15)
        main_sizer.Add(rebase_desc, 0, wx.LEFT | wx.RIGHT | wx.BOTTOM, 15)

        # Merge option
        merge_btn = wx.Button(panel, label="Merge", size=(-1, 35))
        merge_desc = wx.StaticText(panel, label=(
            "Create a merge commit combining both histories.\n"
            "Preserves exact commit history."
        ))
        merge_desc.SetForegroundColour(wx.Colour(107, 114, 128))
        merge_btn.Bind(wx.EVT_BUTTON, self._on_merge)
        main_sizer.Add(merge_btn, 0, wx.EXPAND | wx.LEFT | wx.RIGHT, 15)
        main_sizer.Add(merge_desc, 0, wx.LEFT | wx.RIGHT | wx.BOTTOM, 15)

        main_sizer.AddStretchSpacer()

        # Cancel button
        cancel_btn = wx.Button(panel, wx.ID_CANCEL, "Cancel")
        cancel_btn.Bind(wx.EVT_BUTTON, self._on_cancel)
        main_sizer.Add(cancel_btn, 0, wx.ALIGN_CENTER | wx.BOTTOM, 15)

        panel.SetSizer(main_sizer)
        self.Centre()

    def _on_rebase(self, event):
        self.strategy = self.STRATEGY_REBASE
        self.EndModal(wx.ID_OK)

    def _on_merge(self, event):
        self.strategy = self.STRATEGY_MERGE
        self.EndModal(wx.ID_OK)

    def _on_cancel(self, event):
        self.strategy = self.STRATEGY_CANCEL
        self.EndModal(wx.ID_CANCEL)


class RebaseConflictDialog(wx.Dialog):
    """Dialog for resolving rebase conflicts."""

    def __init__(self, parent, git: GitUtils, conflicts: list):
        """
        Initialize rebase conflict dialog.

        Args:
            parent: Parent window
            git: GitUtils instance
            conflicts: List of conflicting file paths
        """
        super().__init__(parent, title="Rebase Conflicts", size=(500, 400))

        self.git = git
        self.conflicts = conflicts[:]
        self.resolved_files = set()
        self.SetMinSize((500, 400))

        panel = wx.Panel(self)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Header
        header = wx.StaticText(panel, label=f"Rebase conflicts detected ({len(conflicts)} file(s))")
        header.SetFont(wx.Font(12, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        main_sizer.Add(header, 0, wx.ALL, 15)

        # Info text
        info = wx.StaticText(panel, label=(
            "The following files have conflicts during rebase.\n"
            "Choose which version to keep for each file."
        ))
        info.SetForegroundColour(wx.Colour(107, 114, 128))
        main_sizer.Add(info, 0, wx.LEFT | wx.RIGHT | wx.BOTTOM, 15)

        # Conflicts list
        self.conflict_panel = wx.ScrolledWindow(panel)
        self.conflict_panel.SetScrollRate(0, 20)
        conflict_sizer = wx.BoxSizer(wx.VERTICAL)

        self.file_items = {}
        for filepath in conflicts:
            item_sizer = self._create_conflict_item(self.conflict_panel, filepath)
            conflict_sizer.Add(item_sizer, 0, wx.EXPAND | wx.BOTTOM, 10)

        self.conflict_panel.SetSizer(conflict_sizer)
        main_sizer.Add(self.conflict_panel, 1, wx.EXPAND | wx.LEFT | wx.RIGHT, 15)

        # Status text
        self.status_text = wx.StaticText(panel, label="")
        self.status_text.SetForegroundColour(wx.Colour(234, 179, 8))
        main_sizer.Add(self.status_text, 0, wx.ALL, 15)

        # Buttons
        btn_sizer = wx.BoxSizer(wx.HORIZONTAL)
        self.continue_btn = wx.Button(panel, wx.ID_OK, "Continue Rebase")
        self.continue_btn.Enable(False)
        abort_btn = wx.Button(panel, wx.ID_CANCEL, "Abort Rebase")

        btn_sizer.Add(abort_btn, 0, wx.RIGHT, 10)
        btn_sizer.Add(self.continue_btn, 0)

        main_sizer.Add(btn_sizer, 0, wx.ALIGN_CENTER | wx.BOTTOM, 15)

        panel.SetSizer(main_sizer)
        self.Centre()

        self._update_status()

    def _create_conflict_item(self, parent, filepath: str) -> wx.BoxSizer:
        """Create a conflict item with resolution buttons."""
        item_sizer = wx.BoxSizer(wx.HORIZONTAL)

        filename = os.path.basename(filepath)
        file_label = wx.StaticText(parent, label=filename)
        file_label.SetToolTip(filepath)
        item_sizer.Add(file_label, 1, wx.ALIGN_CENTER_VERTICAL)

        ours_btn = wx.Button(parent, label="Keep Ours", size=(90, -1))
        theirs_btn = wx.Button(parent, label="Keep Theirs", size=(90, -1))

        ours_btn.Bind(wx.EVT_BUTTON, lambda e, f=filepath: self._resolve_conflict(f, 'ours'))
        theirs_btn.Bind(wx.EVT_BUTTON, lambda e, f=filepath: self._resolve_conflict(f, 'theirs'))

        item_sizer.Add(ours_btn, 0, wx.RIGHT, 5)
        item_sizer.Add(theirs_btn, 0)

        self.file_items[filepath] = {
            'label': file_label,
            'ours_btn': ours_btn,
            'theirs_btn': theirs_btn,
            'resolved': False,
            'choice': None
        }

        return item_sizer

    def _resolve_conflict(self, filepath: str, choice: str):
        """Resolve a single conflict."""
        try:
            self.git.resolve_conflict(filepath, choice)
            self.resolved_files.add(filepath)

            item = self.file_items[filepath]
            item['resolved'] = True
            item['choice'] = choice
            item['ours_btn'].Enable(False)
            item['theirs_btn'].Enable(False)
            item['label'].SetForegroundColour(wx.Colour(34, 197, 94))
            choice_text = "ours" if choice == 'ours' else "theirs"
            item['label'].SetLabel(f"{os.path.basename(filepath)} (kept {choice_text})")

            self._update_status()
            self.conflict_panel.Refresh()

        except GitError as e:
            wx.MessageBox(f"Failed to resolve conflict: {e}", "Error", wx.OK | wx.ICON_ERROR)

    def _update_status(self):
        """Update status text and button state."""
        resolved_count = len(self.resolved_files)
        total_count = len(self.conflicts)

        if resolved_count == total_count:
            self.status_text.SetLabel(f"All {total_count} conflict(s) resolved. Ready to continue.")
            self.status_text.SetForegroundColour(wx.Colour(34, 197, 94))
            self.continue_btn.Enable(True)
        else:
            remaining = total_count - resolved_count
            self.status_text.SetLabel(f"{remaining} conflict(s) remaining to resolve.")
            self.status_text.SetForegroundColour(wx.Colour(234, 179, 8))
            self.continue_btn.Enable(False)


class BranchDialog(wx.Dialog):
    """Dialog for branch management."""

    def __init__(self, parent, git: GitUtils, mode: str = "switch"):
        """
        Initialize branch dialog.

        Args:
            parent: Parent window
            git: GitUtils instance
            mode: "switch" for switching branches, "create" for creating new branch,
                  "merge" for merging a branch into current
        """
        if mode == "create":
            title = "Create New Branch"
        elif mode == "merge":
            title = "Merge Branch"
        else:
            title = "Switch Branch"
        super().__init__(parent, title=title, size=(400, 250))

        self.git = git
        self.mode = mode
        self.SetMinSize((400, 250))

        panel = wx.Panel(self)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        if mode == "create":
            # New branch name input
            name_sizer = wx.BoxSizer(wx.HORIZONTAL)
            name_label = wx.StaticText(panel, label="Branch name:")
            self.name_input = wx.TextCtrl(panel, size=(250, -1))
            self.name_input.SetHint("feature/my-new-feature")
            name_sizer.Add(name_label, 0, wx.ALIGN_CENTER_VERTICAL | wx.RIGHT, 10)
            name_sizer.Add(self.name_input, 1, wx.EXPAND)
            main_sizer.Add(name_sizer, 0, wx.EXPAND | wx.ALL, 15)

            # Option to switch to new branch
            self.switch_checkbox = wx.CheckBox(panel, label="Switch to new branch after creation")
            self.switch_checkbox.SetValue(True)
            main_sizer.Add(self.switch_checkbox, 0, wx.LEFT | wx.RIGHT, 15)
        else:
            # Branch selection
            branches, current = self.git.get_branches()
            self.current_branch = current

            # For merge mode, filter out the current branch (can't merge into itself)
            if mode == "merge":
                branches = [b for b in branches if b != current]

            branch_sizer = wx.BoxSizer(wx.HORIZONTAL)
            label_text = "Merge from:" if mode == "merge" else "Select branch:"
            branch_label = wx.StaticText(panel, label=label_text)
            self.branch_choice = wx.Choice(panel, choices=branches)

            if mode != "merge" and current in branches:
                self.branch_choice.SetSelection(branches.index(current))
            elif branches:
                self.branch_choice.SetSelection(0)

            branch_sizer.Add(branch_label, 0, wx.ALIGN_CENTER_VERTICAL | wx.RIGHT, 10)
            branch_sizer.Add(self.branch_choice, 1, wx.EXPAND)
            main_sizer.Add(branch_sizer, 0, wx.EXPAND | wx.ALL, 15)

            # Current branch info
            info_text = f"Merging into: {current}" if mode == "merge" else f"Current branch: {current}"
            current_label = wx.StaticText(panel, label=info_text)
            current_label.SetForegroundColour(wx.Colour(107, 114, 128))
            main_sizer.Add(current_label, 0, wx.LEFT | wx.RIGHT, 15)

        # Buttons
        btn_sizer = wx.BoxSizer(wx.HORIZONTAL)
        if mode == "create":
            btn_text = "Create"
        elif mode == "merge":
            btn_text = "Merge"
        else:
            btn_text = "Switch"
        ok_btn = wx.Button(panel, wx.ID_OK, btn_text)
        cancel_btn = wx.Button(panel, wx.ID_CANCEL, "Cancel")

        ok_btn.Bind(wx.EVT_BUTTON, self._on_ok)

        btn_sizer.Add(cancel_btn, 0, wx.RIGHT, 10)
        btn_sizer.Add(ok_btn, 0)

        main_sizer.AddStretchSpacer()
        main_sizer.Add(btn_sizer, 0, wx.ALIGN_CENTER | wx.ALL, 15)

        panel.SetSizer(main_sizer)
        self.Centre()

    def _on_ok(self, event):
        """Handle OK button."""
        try:
            if self.mode == "create":
                name = self.name_input.GetValue().strip()
                if not name:
                    wx.MessageBox("Please enter a branch name.", "Validation Error", wx.OK | wx.ICON_WARNING)
                    return

                # Validate branch name (basic validation)
                if ' ' in name or '..' in name:
                    wx.MessageBox("Invalid branch name.", "Validation Error", wx.OK | wx.ICON_WARNING)
                    return

                switch = self.switch_checkbox.GetValue()
                self.git.create_branch(name, switch=switch)
                self.result_branch = name
                self.result_switched = switch
            elif self.mode == "merge":
                # Just store the selected branch, caller handles the merge
                selection = self.branch_choice.GetSelection()
                if selection == wx.NOT_FOUND:
                    wx.MessageBox("Please select a branch to merge.", "Validation Error", wx.OK | wx.ICON_WARNING)
                    return
                branch = self.branch_choice.GetString(selection)
                self.result_branch = branch
            else:
                # Check if rebase or merge is in progress
                if self.git.is_rebasing():
                    result = wx.MessageBox(
                        "A rebase is in progress. You must abort it before switching branches.\n\n"
                        "Abort the rebase and switch branch?",
                        "Rebase in Progress",
                        wx.YES_NO | wx.ICON_WARNING
                    )
                    if result == wx.YES:
                        self.git.abort_rebase()
                    else:
                        return

                if self.git.is_merging():
                    result = wx.MessageBox(
                        "A merge is in progress. You must abort it before switching branches.\n\n"
                        "Abort the merge and switch branch?",
                        "Merge in Progress",
                        wx.YES_NO | wx.ICON_WARNING
                    )
                    if result == wx.YES:
                        self.git.abort_merge()
                    else:
                        return

                branch = self.branch_choice.GetString(self.branch_choice.GetSelection())
                self.git.switch_branch(branch)
                self.result_branch = branch

            self.EndModal(wx.ID_OK)
        except GitError as e:
            wx.MessageBox(f"Operation failed: {e}", "Error", wx.OK | wx.ICON_ERROR)


class CreateRepositoryDialog(wx.Dialog):
    """Dialog for creating a new repository on Paplix/Gitea."""

    def __init__(self, parent, config: Config, project_config: ProjectConfig):
        super().__init__(parent, title="Create Repository", size=(450, 350))

        self.config = config
        self.project_config = project_config
        self.created_repo: Optional[Repository] = None
        self.SetMinSize((450, 350))

        panel = wx.Panel(self)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Title
        title = wx.StaticText(panel, label="Create New Repository")
        title.SetFont(wx.Font(12, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        main_sizer.Add(title, 0, wx.ALL | wx.ALIGN_CENTER, 15)

        # Repository name
        name_sizer = wx.BoxSizer(wx.HORIZONTAL)
        name_label = wx.StaticText(panel, label="Name:")
        self.name_input = wx.TextCtrl(panel, size=(300, -1))
        self.name_input.SetHint("my-kicad-project")

        # Pre-fill with sanitized project folder name
        project_name = self.project_config._project_dir.name
        # Replace invalid characters with hyphen (valid: a-zA-Z0-9_-)
        sanitized_name = re.sub(r'[^a-zA-Z0-9_-]', '-', project_name)
        # Collapse multiple hyphens and remove leading/trailing hyphens
        sanitized_name = re.sub(r'-+', '-', sanitized_name).strip('-')
        if sanitized_name:
            self.name_input.SetValue(sanitized_name)

        name_sizer.Add(name_label, 0, wx.ALIGN_CENTER_VERTICAL | wx.RIGHT, 10)
        name_sizer.Add(self.name_input, 1, wx.EXPAND)
        main_sizer.Add(name_sizer, 0, wx.EXPAND | wx.LEFT | wx.RIGHT, 15)

        # Description
        desc_sizer = wx.BoxSizer(wx.HORIZONTAL)
        desc_label = wx.StaticText(panel, label="Description:")
        self.desc_input = wx.TextCtrl(panel, size=(300, -1))
        self.desc_input.SetHint("Optional description")
        desc_sizer.Add(desc_label, 0, wx.ALIGN_CENTER_VERTICAL | wx.RIGHT, 10)
        desc_sizer.Add(self.desc_input, 1, wx.EXPAND)
        main_sizer.Add(desc_sizer, 0, wx.EXPAND | wx.ALL, 15)

        # Private checkbox
        self.private_checkbox = wx.CheckBox(panel, label="Private repository")
        self.private_checkbox.SetValue(True)
        main_sizer.Add(self.private_checkbox, 0, wx.LEFT | wx.RIGHT, 15)

        # Info text
        info_text = wx.StaticText(panel, label=(
            "The repository will be created in your organization.\n"
            "You can then set it as the remote origin for this project."
        ))
        info_text.SetForegroundColour(wx.Colour(107, 114, 128))
        main_sizer.Add(info_text, 0, wx.ALL, 15)

        # Buttons
        btn_sizer = wx.BoxSizer(wx.HORIZONTAL)
        create_btn = wx.Button(panel, wx.ID_OK, "Create")
        cancel_btn = wx.Button(panel, wx.ID_CANCEL, "Cancel")

        create_btn.Bind(wx.EVT_BUTTON, self._on_create)

        btn_sizer.Add(cancel_btn, 0, wx.RIGHT, 10)
        btn_sizer.Add(create_btn, 0)

        main_sizer.AddStretchSpacer()
        main_sizer.Add(btn_sizer, 0, wx.ALIGN_CENTER | wx.ALL, 15)

        panel.SetSizer(main_sizer)
        self.Centre()

    def _on_create(self, event):
        """Create the repository."""
        name = self.name_input.GetValue().strip()
        if not name:
            wx.MessageBox("Please enter a repository name.", "Validation Error", wx.OK | wx.ICON_WARNING)
            return

        # Validate name (alphanumeric, hyphens, underscores)
        if not re.match(r'^[a-zA-Z0-9_-]+$', name):
            wx.MessageBox(
                "Repository name can only contain letters, numbers, hyphens, and underscores.",
                "Validation Error",
                wx.OK | wx.ICON_WARNING
            )
            return

        try:
            api = PaplixAPIClient(self.config.base_url, self.config.access_token)
            self.created_repo = api.create_repository(
                name=name,
                description=self.desc_input.GetValue().strip() or None,
                private=self.private_checkbox.GetValue(),
                default_branch=self.project_config.default_branch
            )
            wx.MessageBox(
                f"Repository '{name}' created successfully!\n\nClone URL: {self.created_repo.clone_url}",
                "Success",
                wx.OK | wx.ICON_INFORMATION
            )
            self.EndModal(wx.ID_OK)
        except RepositoryExistsError:
            wx.MessageBox(f"Repository '{name}' already exists.", "Error", wx.OK | wx.ICON_ERROR)
        except AuthenticationError:
            wx.MessageBox("Authentication failed. Please login again.", "Error", wx.OK | wx.ICON_ERROR)
        except APIError as e:
            wx.MessageBox(f"Failed to create repository: {e}", "Error", wx.OK | wx.ICON_ERROR)


class SelectRepositoryDialog(wx.Dialog):
    """Dialog for selecting an existing repository."""

    def __init__(self, parent, config: Config):
        super().__init__(parent, title="Select Repository", size=(500, 400))

        self.config = config
        self.selected_repo: Optional[Repository] = None
        self.repositories: List[Repository] = []
        self.SetMinSize((500, 400))

        panel = wx.Panel(self)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Title
        title = wx.StaticText(panel, label="Select Repository")
        title.SetFont(wx.Font(12, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        main_sizer.Add(title, 0, wx.ALL | wx.ALIGN_CENTER, 15)

        # Repository list
        self.repo_list = wx.ListCtrl(panel, style=wx.LC_REPORT | wx.LC_SINGLE_SEL)
        self.repo_list.InsertColumn(0, "Name", width=150)
        self.repo_list.InsertColumn(1, "Description", width=200)
        self.repo_list.InsertColumn(2, "Private", width=60)
        main_sizer.Add(self.repo_list, 1, wx.EXPAND | wx.LEFT | wx.RIGHT, 15)

        # Refresh button
        refresh_btn = wx.Button(panel, label="Refresh")
        refresh_btn.Bind(wx.EVT_BUTTON, self._on_refresh)
        main_sizer.Add(refresh_btn, 0, wx.ALIGN_RIGHT | wx.ALL, 10)

        # Buttons
        btn_sizer = wx.BoxSizer(wx.HORIZONTAL)
        select_btn = wx.Button(panel, wx.ID_OK, "Select")
        cancel_btn = wx.Button(panel, wx.ID_CANCEL, "Cancel")

        select_btn.Bind(wx.EVT_BUTTON, self._on_select)

        btn_sizer.Add(cancel_btn, 0, wx.RIGHT, 10)
        btn_sizer.Add(select_btn, 0)

        main_sizer.Add(btn_sizer, 0, wx.ALIGN_CENTER | wx.ALL, 15)

        panel.SetSizer(main_sizer)
        self.Centre()

        # Load repositories
        self._load_repositories()

    def _load_repositories(self):
        """Load repositories from API."""
        self.repo_list.DeleteAllItems()

        try:
            api = PaplixAPIClient(self.config.base_url, self.config.access_token)
            self.repositories = api.list_repositories()

            for repo in self.repositories:
                index = self.repo_list.InsertItem(self.repo_list.GetItemCount(), repo.name)
                self.repo_list.SetItem(index, 1, repo.description or "")
                self.repo_list.SetItem(index, 2, "Yes" if repo.private else "No")

        except AuthenticationError:
            wx.MessageBox("Authentication failed. Please login again.", "Error", wx.OK | wx.ICON_ERROR)
        except APIError as e:
            wx.MessageBox(f"Failed to load repositories: {e}", "Error", wx.OK | wx.ICON_ERROR)

    def _on_refresh(self, event):
        """Refresh repository list."""
        self._load_repositories()

    def _on_select(self, event):
        """Select the highlighted repository."""
        selection = self.repo_list.GetFirstSelected()
        if selection == -1:
            wx.MessageBox("Please select a repository.", "Validation Error", wx.OK | wx.ICON_WARNING)
            return

        self.selected_repo = self.repositories[selection]
        self.EndModal(wx.ID_OK)


class CloneRepositoryDialog(wx.Dialog):
    """Dialog for cloning a repository from Paplix."""

    def __init__(self, parent, config: Config):
        super().__init__(parent, title="Clone Repository", size=(550, 500))

        self.config = config
        self.repositories: List[Repository] = []
        self.SetMinSize((550, 500))

        panel = wx.Panel(self)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Title
        title = wx.StaticText(panel, label="Clone Repository from Paplix")
        title.SetFont(wx.Font(12, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        main_sizer.Add(title, 0, wx.ALL | wx.ALIGN_CENTER, 15)

        # Repository list
        repo_label = wx.StaticText(panel, label="Select repository to clone:")
        main_sizer.Add(repo_label, 0, wx.LEFT | wx.RIGHT, 15)

        self.repo_list = wx.ListCtrl(panel, style=wx.LC_REPORT | wx.LC_SINGLE_SEL)
        self.repo_list.InsertColumn(0, "Name", width=150)
        self.repo_list.InsertColumn(1, "Description", width=250)
        self.repo_list.InsertColumn(2, "Private", width=60)
        main_sizer.Add(self.repo_list, 1, wx.EXPAND | wx.ALL, 15)

        # Refresh button
        refresh_btn = wx.Button(panel, label="Refresh")
        refresh_btn.Bind(wx.EVT_BUTTON, self._on_refresh)
        main_sizer.Add(refresh_btn, 0, wx.ALIGN_RIGHT | wx.RIGHT, 15)

        # Destination directory
        dest_label = wx.StaticText(panel, label="Clone to directory:")
        main_sizer.Add(dest_label, 0, wx.LEFT | wx.RIGHT | wx.TOP, 15)

        dest_sizer = wx.BoxSizer(wx.HORIZONTAL)
        self.dest_input = wx.TextCtrl(panel, size=(350, -1))
        self.dest_input.SetHint("Select destination folder...")
        browse_btn = wx.Button(panel, label="Browse...")
        browse_btn.Bind(wx.EVT_BUTTON, self._on_browse)
        dest_sizer.Add(self.dest_input, 1, wx.EXPAND | wx.RIGHT, 5)
        dest_sizer.Add(browse_btn, 0)
        main_sizer.Add(dest_sizer, 0, wx.EXPAND | wx.LEFT | wx.RIGHT | wx.BOTTOM, 15)

        # Subfolder option
        self.create_subfolder = wx.CheckBox(panel, label="Create subfolder with repository name")
        self.create_subfolder.SetValue(True)
        main_sizer.Add(self.create_subfolder, 0, wx.LEFT | wx.RIGHT, 15)

        # Buttons
        btn_sizer = wx.BoxSizer(wx.HORIZONTAL)
        clone_btn = wx.Button(panel, wx.ID_OK, "Clone")
        cancel_btn = wx.Button(panel, wx.ID_CANCEL, "Cancel")

        clone_btn.Bind(wx.EVT_BUTTON, self._on_clone)

        btn_sizer.Add(cancel_btn, 0, wx.RIGHT, 10)
        btn_sizer.Add(clone_btn, 0)

        main_sizer.Add(btn_sizer, 0, wx.ALIGN_CENTER | wx.ALL, 15)

        panel.SetSizer(main_sizer)
        self.Centre()

        # Load repositories
        self._load_repositories()

    def _load_repositories(self):
        """Load repositories from API."""
        self.repo_list.DeleteAllItems()

        try:
            api = PaplixAPIClient(self.config.base_url, self.config.access_token)
            self.repositories = api.list_repositories()

            for repo in self.repositories:
                index = self.repo_list.InsertItem(self.repo_list.GetItemCount(), repo.name)
                self.repo_list.SetItem(index, 1, repo.description or "")
                self.repo_list.SetItem(index, 2, "Yes" if repo.private else "No")

        except AuthenticationError:
            wx.MessageBox("Authentication failed. Please login again.", "Error", wx.OK | wx.ICON_ERROR)
        except APIError as e:
            wx.MessageBox(f"Failed to load repositories: {e}", "Error", wx.OK | wx.ICON_ERROR)

    def _on_refresh(self, event):
        """Refresh repository list."""
        self._load_repositories()

    def _on_browse(self, event):
        """Open directory picker."""
        dlg = wx.DirDialog(
            self,
            "Select destination folder",
            style=wx.DD_DEFAULT_STYLE | wx.DD_DIR_MUST_EXIST
        )
        if dlg.ShowModal() == wx.ID_OK:
            self.dest_input.SetValue(dlg.GetPath())
        dlg.Destroy()

    def _on_clone(self, event):
        """Clone the selected repository."""
        # Validate selection
        selection = self.repo_list.GetFirstSelected()
        if selection == -1:
            wx.MessageBox("Please select a repository to clone.", "Validation Error", wx.OK | wx.ICON_WARNING)
            return

        # Validate destination
        dest_path = self.dest_input.GetValue().strip()
        if not dest_path:
            wx.MessageBox("Please select a destination folder.", "Validation Error", wx.OK | wx.ICON_WARNING)
            return

        if not os.path.isdir(dest_path):
            wx.MessageBox("Destination folder does not exist.", "Validation Error", wx.OK | wx.ICON_WARNING)
            return

        repo = self.repositories[selection]

        # Build final destination path
        if self.create_subfolder.GetValue():
            final_path = os.path.join(dest_path, repo.name)
        else:
            final_path = dest_path

        # Check if destination already exists
        if os.path.exists(final_path) and os.listdir(final_path):
            wx.MessageBox(
                f"Destination folder already exists and is not empty:\n{final_path}",
                "Error",
                wx.OK | wx.ICON_ERROR
            )
            return

        # Get authenticated URL
        auth_url = repo.get_authenticated_url(self.config.gitea_access_token)

        # Perform clone
        try:
            wx.BeginBusyCursor()
            result = GitUtils.clone(auth_url, final_path)
            wx.EndBusyCursor()

            if result.warning:
                wx.MessageBox(
                    f"Repository cloned successfully!\n\nLocation: {final_path}\n\nWarning: {result.warning}",
                    "Clone Successful (with warning)",
                    wx.OK | wx.ICON_WARNING
                )
            else:
                wx.MessageBox(
                    f"Repository cloned successfully!\n\nLocation: {final_path}",
                    "Clone Successful",
                    wx.OK | wx.ICON_INFORMATION
                )
            self.EndModal(wx.ID_OK)

        except GitError as e:
            wx.EndBusyCursor()
            wx.MessageBox(f"Clone failed: {e}", "Error", wx.OK | wx.ICON_ERROR)


class RemoteManagementDialog(wx.Dialog):
    """Dialog for managing Git remotes."""

    def __init__(self, parent, git: GitUtils, config: Config, project_config: ProjectConfig):
        super().__init__(parent, title="Manage Remotes", size=(550, 400))

        self.git = git
        self.config = config
        self.project_config = project_config
        self.SetMinSize((550, 400))

        panel = wx.Panel(self)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Title
        title = wx.StaticText(panel, label="Remote Repositories")
        title.SetFont(wx.Font(12, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        main_sizer.Add(title, 0, wx.ALL | wx.ALIGN_CENTER, 15)

        # Remote list
        self.remote_list = wx.ListCtrl(panel, style=wx.LC_REPORT | wx.LC_SINGLE_SEL)
        self.remote_list.InsertColumn(0, "Name", width=80)
        self.remote_list.InsertColumn(1, "Project", width=150)
        self.remote_list.InsertColumn(2, "URL", width=250)
        main_sizer.Add(self.remote_list, 1, wx.EXPAND | wx.LEFT | wx.RIGHT, 15)

        # Action buttons
        action_sizer = wx.BoxSizer(wx.HORIZONTAL)

        add_btn = wx.Button(panel, label="Add Remote")
        add_from_paplix_btn = wx.Button(panel, label="Add from Paplix")
        edit_btn = wx.Button(panel, label="Edit")
        remove_btn = wx.Button(panel, label="Remove")

        add_btn.Bind(wx.EVT_BUTTON, self._on_add)
        add_from_paplix_btn.Bind(wx.EVT_BUTTON, self._on_add_from_paplix)
        edit_btn.Bind(wx.EVT_BUTTON, self._on_edit)
        remove_btn.Bind(wx.EVT_BUTTON, self._on_remove)

        action_sizer.Add(add_btn, 0, wx.RIGHT, 5)
        action_sizer.Add(add_from_paplix_btn, 0, wx.RIGHT, 5)
        action_sizer.Add(edit_btn, 0, wx.RIGHT, 5)
        action_sizer.Add(remove_btn, 0)

        main_sizer.Add(action_sizer, 0, wx.ALL, 15)

        # Close button
        close_btn = wx.Button(panel, wx.ID_CLOSE, "Close")
        close_btn.Bind(wx.EVT_BUTTON, lambda e: self.EndModal(wx.ID_CLOSE))
        main_sizer.Add(close_btn, 0, wx.ALIGN_CENTER | wx.BOTTOM, 15)

        panel.SetSizer(main_sizer)
        self.Centre()

        # Load remotes
        self._load_remotes()

    def _load_remotes(self):
        """Load current remotes."""
        self.remote_list.DeleteAllItems()

        try:
            remotes = self.git.get_remotes()
            for name, url in remotes:
                index = self.remote_list.InsertItem(self.remote_list.GetItemCount(), name)
                # Extract project name and clean URL
                project_name = self._extract_project_name(url)
                clean_url = self._get_clean_url(url)
                self.remote_list.SetItem(index, 1, project_name)
                self.remote_list.SetItem(index, 2, clean_url)
        except GitError as e:
            wx.MessageBox(f"Failed to load remotes: {e}", "Error", wx.OK | wx.ICON_ERROR)

    def _extract_project_name(self, url: str) -> str:
        """Extract project/repository name from Git URL."""
        import re
        # Remove .git suffix if present
        url = re.sub(r'\.git$', '', url)
        # Extract the last path component (repo name)
        # Handle URLs like: https://token@host/org/repo or https://host/org/repo
        match = re.search(r'/([^/]+)$', url)
        if match:
            return match.group(1)
        return url

    def _get_clean_url(self, url: str) -> str:
        """Get a clean URL without authentication token."""
        import re
        # Remove token from URL: https://token@host/path -> https://host/path
        clean = re.sub(r'(https?://)([^@]+)@', r'\1', url)
        # Remove .git suffix for cleaner display
        clean = re.sub(r'\.git$', '', clean)
        return clean

    def _on_add(self, event):
        """Add a new remote manually."""
        dlg = AddRemoteDialog(self)
        if dlg.ShowModal() == wx.ID_OK:
            name = dlg.name_input.GetValue().strip()
            url = dlg.url_input.GetValue().strip()
            try:
                self.git.set_remote(url, name)
                self._load_remotes()
            except GitError as e:
                wx.MessageBox(f"Failed to add remote: {e}", "Error", wx.OK | wx.ICON_ERROR)
        dlg.Destroy()

    def _on_add_from_paplix(self, event):
        """Add remote from Paplix repository."""
        if not self.config.is_authenticated():
            wx.MessageBox("Please authenticate with Paplix first.", "Authentication Required", wx.OK | wx.ICON_WARNING)
            return

        # First, let user choose: create new or select existing
        choices = ["Create new repository", "Select existing repository"]
        dlg = wx.SingleChoiceDialog(self, "Choose an option:", "Add from Paplix", choices)

        if dlg.ShowModal() == wx.ID_OK:
            selection = dlg.GetSelection()
            dlg.Destroy()

            if selection == 0:
                # Create new repository
                create_dlg = CreateRepositoryDialog(self, self.config, self.project_config)
                if create_dlg.ShowModal() == wx.ID_OK and create_dlg.created_repo:
                    self._add_repo_as_remote(create_dlg.created_repo)
                create_dlg.Destroy()
            else:
                # Select existing repository
                select_dlg = SelectRepositoryDialog(self, self.config)
                if select_dlg.ShowModal() == wx.ID_OK and select_dlg.selected_repo:
                    self._add_repo_as_remote(select_dlg.selected_repo)
                select_dlg.Destroy()
        else:
            dlg.Destroy()

    def _add_repo_as_remote(self, repo: Repository):
        """Add a repository as a remote with authentication."""
        # Ask for remote name
        name_dlg = wx.TextEntryDialog(self, "Enter remote name:", "Remote Name", "origin")
        if name_dlg.ShowModal() == wx.ID_OK:
            remote_name = name_dlg.GetValue().strip() or "origin"
            name_dlg.Destroy()

            # Get authenticated URL (uses clone_url_authenticated from API if available)
            auth_url = repo.get_authenticated_url(self.config.gitea_access_token)

            try:
                self.git.set_remote(auth_url, remote_name)
                wx.MessageBox(
                    f"Remote '{remote_name}' added successfully!\n\nRepository: {repo.name}",
                    "Success",
                    wx.OK | wx.ICON_INFORMATION
                )
                self._load_remotes()
            except GitError as e:
                wx.MessageBox(f"Failed to add remote: {e}", "Error", wx.OK | wx.ICON_ERROR)
        else:
            name_dlg.Destroy()

    def _on_edit(self, event):
        """Edit selected remote."""
        selection = self.remote_list.GetFirstSelected()
        if selection == -1:
            wx.MessageBox("Please select a remote to edit.", "Info", wx.OK | wx.ICON_INFORMATION)
            return

        name = self.remote_list.GetItemText(selection, 0)
        current_url = self.git.get_remote_url(name) or ""

        dlg = AddRemoteDialog(self, name=name, url=current_url, edit_mode=True)
        if dlg.ShowModal() == wx.ID_OK:
            new_url = dlg.url_input.GetValue().strip()
            try:
                self.git.set_remote(new_url, name)
                self._load_remotes()
            except GitError as e:
                wx.MessageBox(f"Failed to update remote: {e}", "Error", wx.OK | wx.ICON_ERROR)
        dlg.Destroy()

    def _on_remove(self, event):
        """Remove selected remote."""
        selection = self.remote_list.GetFirstSelected()
        if selection == -1:
            wx.MessageBox("Please select a remote to remove.", "Info", wx.OK | wx.ICON_INFORMATION)
            return

        name = self.remote_list.GetItemText(selection, 0)

        if wx.MessageBox(
            f"Are you sure you want to remove remote '{name}'?",
            "Confirm Remove",
            wx.YES_NO | wx.ICON_QUESTION
        ) == wx.YES:
            try:
                self.git.remove_remote(name)
                self._load_remotes()
            except GitError as e:
                wx.MessageBox(f"Failed to remove remote: {e}", "Error", wx.OK | wx.ICON_ERROR)


class AddRemoteDialog(wx.Dialog):
    """Dialog for adding/editing a remote."""

    def __init__(self, parent, name: str = "", url: str = "", edit_mode: bool = False):
        title = "Edit Remote" if edit_mode else "Add Remote"
        super().__init__(parent, title=title, size=(450, 200))

        self.edit_mode = edit_mode
        self.SetMinSize((450, 200))

        panel = wx.Panel(self)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Name input
        name_sizer = wx.BoxSizer(wx.HORIZONTAL)
        name_label = wx.StaticText(panel, label="Name:")
        self.name_input = wx.TextCtrl(panel, value=name, size=(300, -1))
        if edit_mode:
            self.name_input.Disable()
        else:
            self.name_input.SetHint("origin")
        name_sizer.Add(name_label, 0, wx.ALIGN_CENTER_VERTICAL | wx.RIGHT, 10)
        name_sizer.Add(self.name_input, 1, wx.EXPAND)
        main_sizer.Add(name_sizer, 0, wx.EXPAND | wx.ALL, 15)

        # URL input
        url_sizer = wx.BoxSizer(wx.HORIZONTAL)
        url_label = wx.StaticText(panel, label="URL:")
        self.url_input = wx.TextCtrl(panel, value=url, size=(300, -1))
        self.url_input.SetHint("https://git.example.com/org/repo.git")
        url_sizer.Add(url_label, 0, wx.ALIGN_CENTER_VERTICAL | wx.RIGHT, 10)
        url_sizer.Add(self.url_input, 1, wx.EXPAND)
        main_sizer.Add(url_sizer, 0, wx.EXPAND | wx.LEFT | wx.RIGHT, 15)

        # Buttons
        btn_sizer = wx.BoxSizer(wx.HORIZONTAL)
        ok_btn = wx.Button(panel, wx.ID_OK, "Save" if edit_mode else "Add")
        cancel_btn = wx.Button(panel, wx.ID_CANCEL, "Cancel")

        ok_btn.Bind(wx.EVT_BUTTON, self._on_ok)

        btn_sizer.Add(cancel_btn, 0, wx.RIGHT, 10)
        btn_sizer.Add(ok_btn, 0)

        main_sizer.AddStretchSpacer()
        main_sizer.Add(btn_sizer, 0, wx.ALIGN_CENTER | wx.ALL, 15)

        panel.SetSizer(main_sizer)
        self.Centre()

    def _on_ok(self, event):
        """Validate and close."""
        if not self.edit_mode and not self.name_input.GetValue().strip():
            wx.MessageBox("Please enter a remote name.", "Validation Error", wx.OK | wx.ICON_WARNING)
            return

        if not self.url_input.GetValue().strip():
            wx.MessageBox("Please enter a URL.", "Validation Error", wx.OK | wx.ICON_WARNING)
            return

        self.EndModal(wx.ID_OK)


class SettingsPanel(wx.Panel):
    """Settings panel for global and project settings."""

    def __init__(self, parent, config: Config, project_config: ProjectConfig,
                 on_settings_changed: Optional[Callable] = None):
        super().__init__(parent)

        self.config = config
        self.project_config = project_config
        self.on_settings_changed = on_settings_changed

        self._create_ui()

    def _create_ui(self):
        """Create the settings panel UI."""
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Global Settings Section
        global_box = wx.StaticBox(self, label="Global Settings")
        global_sizer = wx.StaticBoxSizer(global_box, wx.VERTICAL)

        # Service URL
        url_sizer = wx.BoxSizer(wx.HORIZONTAL)
        url_label = wx.StaticText(self, label="Service URL:")
        self.url_input = wx.TextCtrl(self, size=(300, -1))
        if self.config.base_url:
            self.url_input.SetValue(self.config.base_url)
        else:
            self.url_input.SetHint("https://your-paplix-instance.com")

        url_sizer.Add(url_label, 0, wx.ALIGN_CENTER_VERTICAL | wx.RIGHT, 10)
        url_sizer.Add(self.url_input, 1, wx.EXPAND)
        global_sizer.Add(url_sizer, 0, wx.EXPAND | wx.ALL, 10)

        main_sizer.Add(global_sizer, 0, wx.EXPAND | wx.ALL, 10)

        # Project Settings Section
        project_box = wx.StaticBox(self, label="Project Settings")
        project_sizer = wx.StaticBoxSizer(project_box, wx.VERTICAL)

        # Auto-save checkbox
        self.auto_commit_cb = wx.CheckBox(self, label="Auto-save when plugin opens")
        self.auto_commit_cb.SetValue(self.project_config.auto_commit_on_focus)
        project_sizer.Add(self.auto_commit_cb, 0, wx.ALL, 5)

        # Auto-pull checkbox
        self.auto_pull_cb = wx.CheckBox(self, label="Auto-pull when plugin opens")
        self.auto_pull_cb.SetValue(self.project_config.auto_pull_on_open)
        project_sizer.Add(self.auto_pull_cb, 0, wx.ALL, 5)

        project_sizer.AddSpacer(10)

        # Default commit message
        msg_label = wx.StaticText(self, label="Default commit message:")
        project_sizer.Add(msg_label, 0, wx.LEFT | wx.RIGHT | wx.TOP, 5)

        self.commit_msg_input = wx.TextCtrl(self, size=(-1, -1))
        self.commit_msg_input.SetValue(self.project_config.default_commit_message)
        project_sizer.Add(self.commit_msg_input, 0, wx.EXPAND | wx.LEFT | wx.RIGHT | wx.BOTTOM, 5)

        # Default branch
        branch_sizer = wx.BoxSizer(wx.HORIZONTAL)
        branch_label = wx.StaticText(self, label="Default branch:")
        self.branch_input = wx.TextCtrl(self, size=(150, -1))
        self.branch_input.SetValue(self.project_config.default_branch)
        branch_sizer.Add(branch_label, 0, wx.ALIGN_CENTER_VERTICAL | wx.RIGHT, 10)
        branch_sizer.Add(self.branch_input, 0)
        project_sizer.Add(branch_sizer, 0, wx.ALL, 5)

        main_sizer.Add(project_sizer, 0, wx.EXPAND | wx.LEFT | wx.RIGHT | wx.BOTTOM, 10)

        # Save button
        self.save_btn = wx.Button(self, label="Save Settings")
        self.save_btn.Bind(wx.EVT_BUTTON, self._on_save)
        main_sizer.Add(self.save_btn, 0, wx.ALIGN_CENTER | wx.ALL, 15)

        self.SetSizer(main_sizer)

    def _on_save(self, event):
        """Save all settings."""
        # Save global settings
        url = self.url_input.GetValue().strip()
        if url:
            old_url = self.config.base_url
            self.config.base_url = url
            # If URL changed, clear auth
            if old_url and old_url != url:
                self.config.clear_auth()

        # Save project settings
        self.project_config.save_all(
            auto_commit_on_focus=self.auto_commit_cb.GetValue(),
            auto_push_on_focus=False,  # No longer used
            auto_pull_on_open=self.auto_pull_cb.GetValue(),
            default_commit_message=self.commit_msg_input.GetValue(),
            default_branch=self.branch_input.GetValue()
        )

        if self.on_settings_changed:
            self.on_settings_changed()


class MainDialog(wx.Dialog):
    """Main plugin dialog with all version control operations."""

    def __init__(self, parent, project_path: str):
        super().__init__(parent, title="Paplix Version Control", size=(600, 750))

        self.project_path = project_path
        self.config = Config()
        self.project_config = ProjectConfig(project_path)
        self.git = GitUtils(project_path)

        self.SetMinSize((600, 750))

        self._create_ui()
        self._run_auto_operations()
        self._refresh_status()

    def _create_ui(self):
        """Create the main dialog UI with tabbed interface."""
        panel = wx.Panel(self)
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Title
        title = wx.StaticText(panel, label="Paplix Version Control")
        title.SetFont(wx.Font(16, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_BOLD))
        main_sizer.Add(title, 0, wx.ALL | wx.ALIGN_CENTER, 10)

        # Project path
        path_text = wx.StaticText(panel, label=f"Project: {self.project_path}")
        path_text.SetForegroundColour(wx.Colour(107, 114, 128))
        main_sizer.Add(path_text, 0, wx.LEFT | wx.RIGHT | wx.BOTTOM, 10)

        # Notebook (tabbed interface)
        self.notebook = wx.Notebook(panel)

        # Main Tab
        self.main_tab = wx.Panel(self.notebook)
        self._create_main_tab()
        self.notebook.AddPage(self.main_tab, "Main")

        # Settings Tab
        self.settings_panel = SettingsPanel(
            self.notebook, self.config, self.project_config,
            on_settings_changed=self._on_settings_changed
        )
        self.notebook.AddPage(self.settings_panel, "Settings")

        main_sizer.Add(self.notebook, 1, wx.EXPAND | wx.LEFT | wx.RIGHT, 10)

        # Status message bar (at the bottom)
        self.status_bar = wx.StaticText(panel, label="")
        self.status_bar.SetFont(wx.Font(9, wx.FONTFAMILY_DEFAULT, wx.FONTSTYLE_NORMAL, wx.FONTWEIGHT_NORMAL))
        main_sizer.Add(self.status_bar, 0, wx.EXPAND | wx.ALL, 10)

        # Close button
        close_btn = wx.Button(panel, wx.ID_CLOSE, "Close")
        close_btn.Bind(wx.EVT_BUTTON, lambda e: self.EndModal(wx.ID_CLOSE))
        main_sizer.Add(close_btn, 0, wx.ALIGN_CENTER | wx.BOTTOM, 15)

        panel.SetSizer(main_sizer)
        self.Centre()

    def _create_main_tab(self):
        """Create the main operations tab."""
        main_sizer = wx.BoxSizer(wx.VERTICAL)

        # Authentication panel
        self.auth_panel = AuthPanel(self.main_tab, self.config, self._on_auth_changed)
        main_sizer.Add(self.auth_panel, 0, wx.EXPAND | wx.ALL, 5)

        # Status panel
        self.status_panel = GitStatusPanel(self.main_tab, self.git)
        main_sizer.Add(self.status_panel, 0, wx.EXPAND | wx.ALL, 5)

        # Operations panel
        ops_box = wx.StaticBox(self.main_tab, label="Operations")
        ops_sizer = wx.StaticBoxSizer(ops_box, wx.VERTICAL)

        # Initialize / Clone buttons
        init_sizer = wx.BoxSizer(wx.HORIZONTAL)
        self.init_btn = wx.Button(self.main_tab, label="Initialize Repository")
        self.clone_btn = wx.Button(self.main_tab, label="Clone Repository")
        self.init_btn.Bind(wx.EVT_BUTTON, self._on_init)
        self.clone_btn.Bind(wx.EVT_BUTTON, self._on_clone)
        init_sizer.Add(self.init_btn, 1, wx.EXPAND | wx.RIGHT, 5)
        init_sizer.Add(self.clone_btn, 1, wx.EXPAND)
        ops_sizer.Add(init_sizer, 0, wx.EXPAND | wx.ALL, 5)

        # Commit button
        self.commit_btn = wx.Button(self.main_tab, label="Commit Changes")
        self.commit_btn.Bind(wx.EVT_BUTTON, self._on_commit)
        ops_sizer.Add(self.commit_btn, 0, wx.EXPAND | wx.LEFT | wx.RIGHT | wx.BOTTOM, 5)

        # Push/Pull buttons
        sync_sizer = wx.BoxSizer(wx.HORIZONTAL)
        self.push_btn = wx.Button(self.main_tab, label="Push")
        self.pull_btn = wx.Button(self.main_tab, label="Pull")
        self.push_btn.Bind(wx.EVT_BUTTON, self._on_push)
        self.pull_btn.Bind(wx.EVT_BUTTON, self._on_pull)
        sync_sizer.Add(self.push_btn, 1, wx.EXPAND | wx.RIGHT, 5)
        sync_sizer.Add(self.pull_btn, 1, wx.EXPAND)
        ops_sizer.Add(sync_sizer, 0, wx.EXPAND | wx.LEFT | wx.RIGHT | wx.BOTTOM, 5)

        # Branch buttons
        branch_sizer = wx.BoxSizer(wx.HORIZONTAL)
        self.new_branch_btn = wx.Button(self.main_tab, label="New Branch")
        self.switch_branch_btn = wx.Button(self.main_tab, label="Switch Branch")
        self.new_branch_btn.Bind(wx.EVT_BUTTON, self._on_new_branch)
        self.switch_branch_btn.Bind(wx.EVT_BUTTON, self._on_switch_branch)
        branch_sizer.Add(self.new_branch_btn, 1, wx.EXPAND | wx.RIGHT, 5)
        branch_sizer.Add(self.switch_branch_btn, 1, wx.EXPAND)
        ops_sizer.Add(branch_sizer, 0, wx.EXPAND | wx.LEFT | wx.RIGHT | wx.BOTTOM, 5)

        # Remote management button
        self.manage_remotes_btn = wx.Button(self.main_tab, label="Manage Remotes")
        self.manage_remotes_btn.Bind(wx.EVT_BUTTON, self._on_manage_remotes)
        ops_sizer.Add(self.manage_remotes_btn, 0, wx.EXPAND | wx.LEFT | wx.RIGHT | wx.BOTTOM, 5)

        main_sizer.Add(ops_sizer, 0, wx.EXPAND | wx.ALL, 5)

        self.main_tab.SetSizer(main_sizer)

    def _show_status(self, message: str, status_type: str = STATUS_INFO):
        """Show a status message in the status bar."""
        colors = {
            STATUS_SUCCESS: wx.Colour(34, 197, 94),   # Green
            STATUS_WARNING: wx.Colour(234, 179, 8),   # Yellow
            STATUS_ERROR: wx.Colour(239, 68, 68),     # Red
            STATUS_INFO: wx.Colour(107, 114, 128),    # Gray
        }
        self.status_bar.SetForegroundColour(colors.get(status_type, colors[STATUS_INFO]))
        self.status_bar.SetLabel(message)

    def _run_auto_operations(self):
        """Run automatic operations on dialog open (auto-pull, auto-commit)."""
        status = self.git.get_status()
        if not status.is_repo or not status.has_remote:
            return

        # Auto-pull on open
        if self.project_config.auto_pull_on_open and status.has_upstream:
            try:
                result = self.git.pull()
                if result.warning:
                    self._show_status(f"Auto-pull: {result.warning}", STATUS_WARNING)
                else:
                    self._show_status("Auto-pull completed", STATUS_SUCCESS)
            except GitError as e:
                self._show_status(f"Auto-pull failed: {e}", STATUS_ERROR)
                return  # Don't auto-commit if pull failed

        # Refresh status after pull
        status = self.git.get_status()

        # Auto-save on focus (save files only, no commit)
        if self.project_config.auto_commit_on_focus:
            try:
                board = pcbnew.GetBoard()
                if board and board.GetFileName():
                    pcbnew.SaveBoard(board.GetFileName(), board)
                    self._show_status("Files saved", STATUS_SUCCESS)
            except Exception as e:
                self._show_status(f"Auto-save failed: {e}", STATUS_ERROR)

    def _on_settings_changed(self):
        """Handle settings change."""
        self._show_status("Settings saved", STATUS_SUCCESS)
        self._refresh_status()

    def _refresh_status(self):
        """Refresh the repository status and update UI."""
        status = self.status_panel.refresh()

        # Update button states based on status
        self.init_btn.Enable(not status.is_repo)
        self.clone_btn.Enable(self.config.is_authenticated())  # Clone needs auth, not existing repo
        self.commit_btn.Enable(status.is_repo and status.has_changes)
        self.push_btn.Enable(status.can_push)
        self.pull_btn.Enable(status.is_repo and status.has_remote)
        self.new_branch_btn.Enable(status.is_repo)
        self.switch_branch_btn.Enable(status.is_repo)
        self.manage_remotes_btn.Enable(status.is_repo)

    def _on_auth_changed(self, auth_success=False, ssl_warning=None, logged_out=False, url_changed=False):
        """Handle authentication state change."""
        if auth_success:
            if ssl_warning:
                self._show_status("Authenticated (with SSL warning)", STATUS_WARNING)
            else:
                self._show_status("Successfully authenticated", STATUS_SUCCESS)
        elif logged_out:
            self._show_status("Logged out", STATUS_INFO)
        elif url_changed:
            self._show_status("URL changed - please re-authenticate", STATUS_INFO)
        self._refresh_status()

    def _on_init(self, event):
        """Initialize a new Git repository."""
        if wx.MessageBox(
            "Initialize a new Git repository in this project?\n\n"
            "This will create a .git folder in your project directory.",
            "Initialize Repository",
            wx.YES_NO | wx.ICON_QUESTION
        ) == wx.YES:
            try:
                self.git.init()
                self._show_status("Repository initialized successfully", STATUS_SUCCESS)
                self._refresh_status()
            except GitError as e:
                wx.MessageBox(f"Failed to initialize repository: {e}", "Error", wx.OK | wx.ICON_ERROR)

    def _on_clone(self, event):
        """Open clone repository dialog."""
        if not self.config.is_authenticated():
            wx.MessageBox(
                "Please authenticate with Paplix first.",
                "Authentication Required",
                wx.OK | wx.ICON_WARNING
            )
            return

        dlg = CloneRepositoryDialog(self, self.config)
        dlg.ShowModal()
        dlg.Destroy()

    def _on_commit(self, event):
        """Open commit dialog."""
        status = self.git.get_status()
        if not status.has_changes:
            self._show_status("No changes to commit", STATUS_INFO)
            return

        dlg = CommitDialog(self, self.git, status)
        if dlg.ShowModal() == wx.ID_OK:
            commit_hash = getattr(dlg, 'commit_hash', '')[:8]
            self._show_status(f"Commit created ({commit_hash})", STATUS_SUCCESS)
            self._refresh_status()
        dlg.Destroy()

    def _on_push(self, event):
        """Push changes to remote."""
        self.push_btn.Disable()
        self.push_btn.SetLabel("Pushing...")

        try:
            # Token is embedded in remote URL, no credentials needed
            result = self.git.push()
            if result.warning:
                # Show warning dialog for SSL issues
                wx.MessageBox(
                    f"Push successful!\n\nWarning: {result.warning}",
                    "Success (with warning)",
                    wx.OK | wx.ICON_WARNING
                )
                self._show_status("Push completed (with SSL warning)", STATUS_WARNING)
            else:
                self._show_status("Push successful", STATUS_SUCCESS)
        except GitError as e:
            wx.MessageBox(f"Push failed: {e}", "Error", wx.OK | wx.ICON_ERROR)
        finally:
            self.push_btn.Enable()
            self.push_btn.SetLabel("Push")
            self._refresh_status()

    def _on_pull(self, event):
        """Pull changes from remote."""
        self.pull_btn.Disable()
        self.pull_btn.SetLabel("Pulling...")

        # Get board file hash before pull
        board_path = self._get_board_path()
        old_hash = self.git.get_file_hash(board_path) if board_path else None

        try:
            # Token is embedded in remote URL, no credentials needed
            result = self.git.pull()
            if result.warning:
                # Show warning dialog for SSL issues
                wx.MessageBox(
                    f"Pull successful!\n\nWarning: {result.warning}",
                    "Success (with warning)",
                    wx.OK | wx.ICON_WARNING
                )
                self._show_status("Pull completed (with SSL warning)", STATUS_WARNING)
            else:
                self._show_status("Pull successful", STATUS_SUCCESS)

            # Check if board needs reload
            self._handle_board_reload_if_needed(old_hash, "Pull successful")
        except GitError as e:
            error_msg = str(e)

            # Check if it's a conflict error - offer rebase or merge choice
            if 'CONFLICT' in error_msg or 'conflict' in error_msg.lower():
                status = self.git.get_status()
                dlg = SyncStrategyDialog(self, status.ahead, status.behind)
                modal_result = dlg.ShowModal()
                strategy = dlg.strategy
                dlg.Destroy()

                if modal_result == wx.ID_OK and strategy != SyncStrategyDialog.STRATEGY_CANCEL:
                    if self._sync_with_remote(strategy):
                        self._handle_board_reload_if_needed(old_hash, "Pull successful")
                else:
                    self._show_status("Pull cancelled", STATUS_WARNING)
            else:
                wx.MessageBox(f"Pull failed: {e}", "Error", wx.OK | wx.ICON_ERROR)
        finally:
            self.pull_btn.Enable()
            self.pull_btn.SetLabel("Pull")
            self._refresh_status()

    def _sync_with_remote(self, strategy: str) -> bool:
        """
        Sync local branch with remote using the chosen strategy.

        Args:
            strategy: SyncStrategyDialog.STRATEGY_REBASE or STRATEGY_MERGE

        Returns:
            True if sync succeeded, False otherwise
        """
        if strategy == SyncStrategyDialog.STRATEGY_REBASE:
            self._show_status("Rebasing...", STATUS_INFO)
            result = self.git.pull_rebase()

            if result.has_conflicts:
                dlg = RebaseConflictDialog(self, self.git, result.conflicts)
                modal_result = dlg.ShowModal()
                dlg.Destroy()

                if modal_result == wx.ID_CANCEL:
                    try:
                        self.git.abort_rebase()
                        self._show_status("Rebase aborted", STATUS_WARNING)
                    except GitError:
                        pass
                    return False

                try:
                    continue_result = self.git.continue_rebase()
                    if not continue_result.success:
                        wx.MessageBox(
                            f"Failed to continue rebase: {continue_result.message}",
                            "Error", wx.OK | wx.ICON_ERROR
                        )
                        return False
                except GitError as e:
                    wx.MessageBox(f"Failed to continue rebase: {e}", "Error", wx.OK | wx.ICON_ERROR)
                    return False

            if not result.success and not result.has_conflicts:
                wx.MessageBox(f"Rebase failed: {result.message}", "Error", wx.OK | wx.ICON_ERROR)
                self._show_status("Rebase failed", STATUS_ERROR)
                return False

            self._show_status("Rebase successful", STATUS_SUCCESS)
            return True

        else:  # STRATEGY_MERGE
            self._show_status("Merging...", STATUS_INFO)
            try:
                self.git.pull()

                if self.git.is_merging():
                    conflicts = self.git.get_conflicting_files()
                    if conflicts:
                        dlg = MergeConflictDialog(self, self.git, conflicts)
                        modal_result = dlg.ShowModal()
                        dlg.Destroy()

                        if modal_result == wx.ID_CANCEL:
                            try:
                                self.git.abort_merge()
                                self._show_status("Merge aborted", STATUS_WARNING)
                            except GitError:
                                pass
                            return False

                self._show_status("Merge successful", STATUS_SUCCESS)
                return True

            except GitError as e:
                error_msg = str(e)
                if 'CONFLICT' in error_msg or 'conflict' in error_msg.lower():
                    conflicts = self.git.get_conflicting_files()
                    if conflicts:
                        dlg = MergeConflictDialog(self, self.git, conflicts)
                        modal_result = dlg.ShowModal()
                        dlg.Destroy()

                        if modal_result == wx.ID_CANCEL:
                            try:
                                self.git.abort_merge()
                                self._show_status("Merge aborted", STATUS_WARNING)
                            except GitError:
                                pass
                            return False

                        self._show_status("Merge successful", STATUS_SUCCESS)
                        return True

                wx.MessageBox(f"Merge failed: {e}", "Error", wx.OK | wx.ICON_ERROR)
                self._show_status("Merge failed", STATUS_ERROR)
                return False

    def _on_new_branch(self, event):
        """Open new branch dialog."""
        dlg = BranchDialog(self, self.git, mode="create")
        if dlg.ShowModal() == wx.ID_OK:
            branch_name = getattr(dlg, 'result_branch', '')
            switched = getattr(dlg, 'result_switched', False)
            if switched:
                self._show_status(f"Branch '{branch_name}' created and switched", STATUS_SUCCESS)
            else:
                self._show_status(f"Branch '{branch_name}' created", STATUS_SUCCESS)
            self._refresh_status()
        dlg.Destroy()

    def _on_switch_branch(self, event):
        """Open switch branch dialog."""
        # Get board file hash before switch
        board_path = self._get_board_path()
        old_hash = self.git.get_file_hash(board_path) if board_path else None

        dlg = BranchDialog(self, self.git, mode="switch")
        if dlg.ShowModal() == wx.ID_OK:
            branch_name = getattr(dlg, 'result_branch', '')
            # Check if board needs reload
            if not self._handle_board_reload_if_needed(old_hash, f"Switched to branch '{branch_name}'"):
                pass  # Status already shown by handler
            else:
                self._show_status(f"Switched to branch '{branch_name}'", STATUS_SUCCESS)
            self._refresh_status()
        dlg.Destroy()

    def _on_manage_remotes(self, event):
        """Open remote management dialog."""
        dlg = RemoteManagementDialog(self, self.git, self.config, self.project_config)
        dlg.ShowModal()
        dlg.Destroy()
        self._refresh_status()
