"""
Git utility functions for Paplix Version Control Plugin.

Provides a simplified interface to Git operations without requiring
users to understand Git commands.
"""

import glob
import os
import subprocess
import shutil
from dataclasses import dataclass
from enum import Enum
from pathlib import Path
from typing import List, Optional, Tuple


class GitError(Exception):
    """Exception raised for Git operation errors."""
    pass


class GitNotFoundError(GitError):
    """Exception raised when Git is not installed."""
    pass


@dataclass
class GitStatus:
    """Represents the status of a Git repository."""
    is_repo: bool = False
    branch: str = ""
    has_changes: bool = False
    staged_files: List[str] = None
    modified_files: List[str] = None
    untracked_files: List[str] = None
    has_remote: bool = False
    remote_url: str = ""
    ahead: int = 0
    behind: int = 0
    has_upstream: bool = False  # True if current branch has upstream tracking
    has_commits: bool = False  # True if there are any commits in the repo

    def __post_init__(self):
        if self.staged_files is None:
            self.staged_files = []
        if self.modified_files is None:
            self.modified_files = []
        if self.untracked_files is None:
            self.untracked_files = []

    @property
    def can_push(self) -> bool:
        """Check if there's anything to push."""
        if not self.is_repo or not self.has_remote:
            return False
        if not self.has_commits:
            return False
        # Can push if: ahead of remote OR no upstream yet (first push)
        return self.ahead > 0 or not self.has_upstream


@dataclass
class CommitInfo:
    """Information about a Git commit."""
    hash: str
    short_hash: str
    author: str
    date: str
    message: str


@dataclass
class GitOperationResult:
    """Result of a git operation that may have SSL warnings."""
    success: bool
    ssl_verified: bool = True
    warning: Optional[str] = None


@dataclass
class MergeResult:
    """Result of a merge operation."""
    success: bool
    conflicts: List[str] = None
    message: str = ""

    def __post_init__(self):
        if self.conflicts is None:
            self.conflicts = []

    @property
    def has_conflicts(self) -> bool:
        return len(self.conflicts) > 0


@dataclass
class RebaseResult:
    """Result of a rebase operation."""
    success: bool
    conflicts: List[str] = None
    message: str = ""

    def __post_init__(self):
        if self.conflicts is None:
            self.conflicts = []

    @property
    def has_conflicts(self) -> bool:
        return len(self.conflicts) > 0


@dataclass
class GraphCommit:
    """Commit information for git graph visualization."""
    hash: str
    short_hash: str
    parents: List[str]  # Parent commit hashes
    refs: List[str]     # Branch/tag names pointing to this commit
    author: str
    date: str
    relative_date: str  # e.g., "2 hours ago"
    message: str

    def __post_init__(self):
        if self.parents is None:
            self.parents = []
        if self.refs is None:
            self.refs = []


@dataclass
class BranchInfo:
    """Information about a git branch."""
    name: str
    is_remote: bool
    is_current: bool
    head_commit: str


# SSL-related error patterns to detect
SSL_ERROR_PATTERNS = [
    'SSL',
    'ssl',
    'certificate',
    'TLS',
    'tls',
    'unable to access',
    'LibreSSL',
    'OpenSSL',
]


def _is_ssl_error(error_message: str) -> bool:
    """Check if an error message is SSL-related."""
    return any(pattern in error_message for pattern in SSL_ERROR_PATTERNS)


class GitUtils:
    """Utility class for Git operations."""

    GIT_DOWNLOAD_URL = "https://git-scm.com/downloads"
    GIT_INSTALL_INSTRUCTIONS = {
        'darwin': "Install Git using Homebrew: brew install git\nOr download from: https://git-scm.com/downloads",
        'linux': "Install Git using your package manager:\n  Ubuntu/Debian: sudo apt install git\n  Fedora: sudo dnf install git\n  Arch: sudo pacman -S git",
        'windows': "Download and install Git from: https://git-scm.com/download/win"
    }

    def __init__(self, working_dir: Optional[str] = None):
        """
        Initialize GitUtils.

        Args:
            working_dir: The working directory for Git operations.
                        If None, uses current working directory.
        """
        self.working_dir = working_dir or os.getcwd()
        self._git_path: Optional[str] = None

    @staticmethod
    def is_git_available() -> bool:
        """Check if Git is installed and available on the system."""
        return shutil.which('git') is not None

    @staticmethod
    def get_git_version() -> Optional[str]:
        """Get the installed Git version."""
        try:
            result = subprocess.run(
                ['git', '--version'],
                capture_output=True,
                text=True,
                timeout=10
            )
            if result.returncode == 0:
                return result.stdout.strip()
        except (subprocess.SubprocessError, FileNotFoundError):
            pass
        return None

    @staticmethod
    def get_install_instructions() -> str:
        """Get platform-specific Git installation instructions."""
        import platform
        system = platform.system().lower()
        if system == 'darwin':
            return GitUtils.GIT_INSTALL_INSTRUCTIONS['darwin']
        elif system == 'linux':
            return GitUtils.GIT_INSTALL_INSTRUCTIONS['linux']
        else:
            return GitUtils.GIT_INSTALL_INSTRUCTIONS['windows']

    def _run_git(self, args: List[str], check: bool = True,
                  disable_ssl_verify: bool = False) -> subprocess.CompletedProcess:
        """
        Run a Git command.

        Args:
            args: Git command arguments (without 'git' prefix)
            check: If True, raise GitError on non-zero exit code
            disable_ssl_verify: If True, disable SSL certificate verification

        Returns:
            CompletedProcess instance

        Raises:
            GitNotFoundError: If Git is not installed
            GitError: If the command fails and check is True
        """
        if not self.is_git_available():
            raise GitNotFoundError("Git is not installed on this system.")

        cmd = ['git'] + args

        # Set up environment
        env = os.environ.copy()
        if disable_ssl_verify:
            env['GIT_SSL_NO_VERIFY'] = 'true'

        try:
            result = subprocess.run(
                cmd,
                cwd=self.working_dir,
                capture_output=True,
                text=True,
                timeout=60,
                env=env
            )
            if check and result.returncode != 0:
                error_msg = result.stderr.strip() or result.stdout.strip()
                raise GitError(f"Git command failed: {error_msg}")
            return result
        except subprocess.TimeoutExpired:
            raise GitError("Git command timed out")
        except FileNotFoundError:
            raise GitNotFoundError("Git executable not found")

    def is_repo(self) -> bool:
        """Check if the working directory is a Git repository."""
        try:
            result = self._run_git(['rev-parse', '--git-dir'], check=False)
            return result.returncode == 0
        except GitNotFoundError:
            return False

    def init(self) -> bool:
        """
        Initialize a new Git repository.

        Returns:
            True if successful
        """
        self._run_git(['init'])
        return True

    def get_status(self) -> GitStatus:
        """
        Get the current repository status.

        Returns:
            GitStatus object with repository information
        """
        status = GitStatus()

        if not self.is_repo():
            return status

        status.is_repo = True

        # Get current branch
        try:
            result = self._run_git(['branch', '--show-current'], check=False)
            if result.returncode == 0:
                status.branch = result.stdout.strip()
        except GitError:
            pass

        # Get file status
        try:
            result = self._run_git(['status', '--porcelain', '-z'], check=False)
            if result.returncode == 0 and result.stdout:
                # Using -z flag: entries are separated by NUL, filenames are not quoted
                entries = result.stdout.split('\0')
                i = 0
                while i < len(entries):
                    entry = entries[i]
                    if not entry or len(entry) < 3:
                        i += 1
                        continue

                    status_code = entry[:2]
                    filename = entry[3:]

                    # Handle renamed files (R status has two filenames)
                    if status_code[0] == 'R':
                        # For renames, the next entry is the original filename
                        # We only care about the new name (current entry)
                        if ' -> ' in filename:
                            filename = filename.split(' -> ')[-1]
                        i += 1  # Skip the next entry (old filename)

                    if status_code[0] in ('A', 'M', 'D', 'R'):
                        status.staged_files.append(filename)
                    if status_code[1] == 'M':
                        status.modified_files.append(filename)
                    if status_code == '??':
                        status.untracked_files.append(filename)

                    i += 1

                status.has_changes = bool(
                    status.staged_files or
                    status.modified_files or
                    status.untracked_files
                )
        except GitError:
            pass

        # Get remote info
        try:
            result = self._run_git(['remote', '-v'], check=False)
            if result.returncode == 0 and result.stdout.strip():
                status.has_remote = True
                lines = result.stdout.strip().split('\n')
                if lines:
                    parts = lines[0].split()
                    if len(parts) >= 2:
                        status.remote_url = parts[1]
        except GitError:
            pass

        # Check if there are any commits
        try:
            result = self._run_git(['rev-parse', 'HEAD'], check=False)
            status.has_commits = result.returncode == 0
        except GitError:
            status.has_commits = False

        # Check if current branch has upstream tracking
        if status.branch:
            try:
                result = self._run_git(
                    ['config', '--get', f'branch.{status.branch}.remote'],
                    check=False
                )
                status.has_upstream = result.returncode == 0 and bool(result.stdout.strip())
            except GitError:
                status.has_upstream = False

        # Get ahead/behind count (only if upstream exists)
        if status.has_remote and status.branch and status.has_upstream:
            try:
                result = self._run_git(
                    ['rev-list', '--left-right', '--count', f'HEAD...origin/{status.branch}'],
                    check=False
                )
                if result.returncode == 0:
                    parts = result.stdout.strip().split()
                    if len(parts) == 2:
                        status.ahead = int(parts[0])
                        status.behind = int(parts[1])
            except (GitError, ValueError):
                pass

        return status

    def add(self, files: Optional[List[str]] = None) -> bool:
        """
        Stage files for commit.

        Args:
            files: List of files to stage. If None, stages all changes.

        Returns:
            True if successful
        """
        if files:
            self._run_git(['add'] + files)
        else:
            self._run_git(['add', '-A'])
        return True

    def commit(self, message: str) -> str:
        """
        Create a commit with the staged changes.

        Args:
            message: Commit message

        Returns:
            The commit hash
        """
        if not message.strip():
            raise GitError("Commit message cannot be empty")

        self._run_git(['commit', '-m', message])

        # Get the commit hash
        result = self._run_git(['rev-parse', 'HEAD'])
        return result.stdout.strip()

    def has_upstream(self, branch: Optional[str] = None) -> bool:
        """
        Check if a branch has an upstream tracking branch configured.

        Args:
            branch: Branch to check. If None, checks current branch.

        Returns:
            True if upstream is configured
        """
        if branch is None:
            # Get current branch
            result = self._run_git(['branch', '--show-current'], check=False)
            if result.returncode != 0 or not result.stdout.strip():
                return False
            branch = result.stdout.strip()

        # Check for upstream
        result = self._run_git(
            ['config', '--get', f'branch.{branch}.remote'],
            check=False
        )
        return result.returncode == 0 and bool(result.stdout.strip())

    def push(self, remote: str = 'origin', branch: Optional[str] = None,
             set_upstream: bool = None, credentials: Optional[Tuple[str, str]] = None) -> GitOperationResult:
        """
        Push commits to remote repository.

        Args:
            remote: Remote name (default: origin)
            branch: Branch to push. If None, pushes current branch.
            set_upstream: If True, sets upstream tracking. If None, auto-detects.
            credentials: Optional (username, password/token) tuple for authentication

        Returns:
            GitOperationResult with success status and SSL warning if applicable
        """
        # Get current branch if not specified
        if branch is None:
            result = self._run_git(['branch', '--show-current'], check=False)
            if result.returncode == 0 and result.stdout.strip():
                branch = result.stdout.strip()

        # Auto-detect if we need to set upstream
        if set_upstream is None:
            set_upstream = not self.has_upstream(branch)

        args = ['push']

        if set_upstream:
            args.append('-u')

        args.append(remote)

        if branch:
            args.append(branch)

        # Set up environment for credentials
        env = os.environ.copy()
        if credentials:
            username, token = credentials
            env['GIT_ASKPASS'] = 'echo'
            env['GIT_USERNAME'] = username
            env['GIT_PASSWORD'] = token

        # First try with SSL verification enabled
        try:
            subprocess.run(
                ['git'] + args,
                cwd=self.working_dir,
                capture_output=True,
                text=True,
                timeout=120,
                env=env,
                check=True
            )
            return GitOperationResult(success=True, ssl_verified=True)
        except subprocess.CalledProcessError as e:
            first_error = e.stderr

            # If SSL error, retry without SSL verification
            if _is_ssl_error(first_error):
                try:
                    subprocess.run(
                        ['git', '-c', 'http.sslVerify=false'] + args,
                        cwd=self.working_dir,
                        capture_output=True,
                        text=True,
                        timeout=120,
                        env=env,
                        check=True
                    )
                    return GitOperationResult(
                        success=True,
                        ssl_verified=False,
                        warning="SSL certificate verification was disabled for this operation. "
                                "The connection may not be secure."
                    )
                except subprocess.CalledProcessError as e2:
                    raise GitError(f"Push failed: {e2.stderr}")

            # Non-SSL error
            raise GitError(f"Push failed: {first_error}")

    def pull(self, remote: str = 'origin', branch: Optional[str] = None,
             credentials: Optional[Tuple[str, str]] = None) -> GitOperationResult:
        """
        Pull changes from remote repository.

        Args:
            remote: Remote name (default: origin)
            branch: Branch to pull. If None, pulls current branch.
            credentials: Optional (username, password/token) tuple for authentication

        Returns:
            GitOperationResult with success status and SSL warning if applicable
        """
        args = ['pull', remote]

        if branch:
            args.append(branch)

        env = os.environ.copy()
        if credentials:
            username, token = credentials
            env['GIT_ASKPASS'] = 'echo'
            env['GIT_USERNAME'] = username
            env['GIT_PASSWORD'] = token

        # First try with SSL verification enabled
        try:
            subprocess.run(
                ['git'] + args,
                cwd=self.working_dir,
                capture_output=True,
                text=True,
                timeout=120,
                env=env,
                check=True
            )
            return GitOperationResult(success=True, ssl_verified=True)
        except subprocess.CalledProcessError as e:
            first_error = e.stderr

            # If SSL error, retry without SSL verification
            if _is_ssl_error(first_error):
                try:
                    subprocess.run(
                        ['git', '-c', 'http.sslVerify=false'] + args,
                        cwd=self.working_dir,
                        capture_output=True,
                        text=True,
                        timeout=120,
                        env=env,
                        check=True
                    )
                    return GitOperationResult(
                        success=True,
                        ssl_verified=False,
                        warning="SSL certificate verification was disabled for this operation. "
                                "The connection may not be secure."
                    )
                except subprocess.CalledProcessError as e2:
                    raise GitError(f"Pull failed: {e2.stderr}")

            # Non-SSL error
            raise GitError(f"Pull failed: {first_error}")

    def pull_rebase(self, remote: str = 'origin', branch: Optional[str] = None,
                    credentials: Optional[Tuple[str, str]] = None) -> RebaseResult:
        """
        Pull changes from remote using rebase strategy.

        Args:
            remote: Remote name (default: origin)
            branch: Branch to pull. If None, pulls current branch.
            credentials: Optional (username, password/token) tuple for authentication

        Returns:
            RebaseResult with success status and list of conflicts if any
        """
        args = ['pull', '--rebase', remote]

        if branch:
            args.append(branch)

        env = os.environ.copy()
        # Prevent git from trying to open an editor in GUI context
        env['GIT_EDITOR'] = 'true'
        env['GIT_TERMINAL_PROMPT'] = '0'
        if credentials:
            username, token = credentials
            env['GIT_ASKPASS'] = 'echo'
            env['GIT_USERNAME'] = username
            env['GIT_PASSWORD'] = token

        # First try with SSL verification enabled
        try:
            result = subprocess.run(
                ['git'] + args,
                cwd=self.working_dir,
                capture_output=True,
                text=True,
                timeout=120,
                env=env
            )

            if result.returncode == 0:
                return RebaseResult(success=True, message="Rebase completed successfully")

            # Check for conflicts
            if 'CONFLICT' in result.stdout or 'CONFLICT' in result.stderr:
                conflicts = self.get_conflicting_files()
                return RebaseResult(
                    success=False,
                    conflicts=conflicts,
                    message=f"Rebase conflicts in {len(conflicts)} file(s)"
                )

            # Other error
            raise GitError(f"Rebase failed: {result.stderr or result.stdout}")

        except subprocess.CalledProcessError as e:
            first_error = e.stderr

            # If SSL error, retry without SSL verification
            if _is_ssl_error(first_error):
                try:
                    result = subprocess.run(
                        ['git', '-c', 'http.sslVerify=false'] + args,
                        cwd=self.working_dir,
                        capture_output=True,
                        text=True,
                        timeout=120,
                        env=env
                    )

                    if result.returncode == 0:
                        return RebaseResult(success=True, message="Rebase completed (SSL warning)")

                    if 'CONFLICT' in result.stdout or 'CONFLICT' in result.stderr:
                        conflicts = self.get_conflicting_files()
                        return RebaseResult(
                            success=False,
                            conflicts=conflicts,
                            message=f"Rebase conflicts in {len(conflicts)} file(s)"
                        )

                    raise GitError(f"Rebase failed: {result.stderr or result.stdout}")
                except subprocess.CalledProcessError as e2:
                    raise GitError(f"Rebase failed: {e2.stderr}")

            raise GitError(f"Rebase failed: {first_error}")

    def get_branches(self) -> Tuple[List[str], str]:
        """
        Get list of branches and current branch.

        Returns:
            Tuple of (list of branch names, current branch name)
        """
        result = self._run_git(['branch'])
        branches = []
        current = ""

        for line in result.stdout.strip().split('\n'):
            if not line:
                continue
            if line.startswith('* '):
                current = line[2:].strip()
                branches.append(current)
            else:
                branches.append(line.strip())

        return branches, current

    def create_branch(self, branch_name: str, switch: bool = True) -> bool:
        """
        Create a new branch.

        Args:
            branch_name: Name for the new branch
            switch: If True, switch to the new branch after creation

        Returns:
            True if successful
        """
        if switch:
            self._run_git(['checkout', '-b', branch_name])
        else:
            self._run_git(['branch', branch_name])
        return True

    def switch_branch(self, branch_name: str) -> bool:
        """
        Switch to an existing branch.

        Args:
            branch_name: Name of the branch to switch to

        Returns:
            True if successful
        """
        # Clean up KiCad lock files before switching (they block git checkout)
        try:
            self._cleanup_lock_files()
        except Exception:
            pass  # Don't let cleanup failure break branch switch
        self._run_git(['switch', branch_name])
        return True

    def _cleanup_lock_files(self):
        """
        Remove KiCad lock files that block git operations.

        KiCad creates .lck files when a board is open. These are untracked
        and prevent branch switching if they exist in the target branch.
        """
        lock_patterns = ['*.lck', '*~*.lck']
        for pattern in lock_patterns:
            for lock_file in glob.glob(os.path.join(self.working_dir, '**', pattern), recursive=True):
                try:
                    os.remove(lock_file)
                except OSError:
                    pass  # Ignore if file can't be deleted (e.g., still in use)

    def merge(self, branch_name: str) -> MergeResult:
        """
        Merge a branch into the current branch.

        Args:
            branch_name: Name of the branch to merge

        Returns:
            MergeResult with success status and list of conflicting files
        """
        # Clean up lock files first
        try:
            self._cleanup_lock_files()
        except Exception:
            pass

        # Attempt merge with --no-commit to handle conflicts manually
        result = self._run_git(['merge', '--no-commit', '--no-ff', branch_name], check=False)

        if result.returncode == 0:
            # Merge succeeded, commit it
            self._run_git(['commit', '-m', f'Merge branch {branch_name}'], check=False)
            return MergeResult(success=True, message=f"Successfully merged '{branch_name}'")

        # Check for conflicts
        if 'CONFLICT' in result.stdout or 'CONFLICT' in result.stderr:
            conflicts = self.get_conflicting_files()
            return MergeResult(
                success=False,
                conflicts=conflicts,
                message=f"Merge conflicts in {len(conflicts)} file(s)"
            )

        # Other error
        error_msg = result.stderr or result.stdout
        raise GitError(f"Merge failed: {error_msg}")

    def get_conflicting_files(self) -> List[str]:
        """
        Get list of files with merge conflicts.

        Returns:
            List of file paths with conflicts
        """
        result = self._run_git(['diff', '--name-only', '--diff-filter=U'], check=False)
        if result.returncode != 0:
            return []

        files = [f.strip() for f in result.stdout.strip().split('\n') if f.strip()]
        return files

    def is_merging(self) -> bool:
        """Check if a merge is in progress."""
        merge_head = Path(self.working_dir) / '.git' / 'MERGE_HEAD'
        return merge_head.exists()

    def is_rebasing(self) -> bool:
        """Check if a rebase is in progress."""
        rebase_merge = Path(self.working_dir) / '.git' / 'rebase-merge'
        rebase_apply = Path(self.working_dir) / '.git' / 'rebase-apply'
        return rebase_merge.exists() or rebase_apply.exists()

    def abort_rebase(self) -> None:
        """Abort an in-progress rebase and restore previous state."""
        self._run_git(['rebase', '--abort'])

    def continue_rebase(self) -> RebaseResult:
        """
        Continue a rebase after resolving conflicts.

        Returns:
            RebaseResult with success status and list of remaining conflicts if any
        """
        # Set environment to prevent git from trying to open an editor
        # This avoids "Terminal is dumb, but EDITOR unset" errors in GUI context
        env = os.environ.copy()
        env['GIT_EDITOR'] = 'true'
        env['GIT_TERMINAL_PROMPT'] = '0'

        result = subprocess.run(
            ['git', 'rebase', '--continue'],
            cwd=self.working_dir,
            capture_output=True,
            text=True,
            timeout=120,
            env=env
        )

        if result.returncode == 0:
            return RebaseResult(success=True, message="Rebase completed")

        # Check for more conflicts
        if 'CONFLICT' in result.stdout or 'CONFLICT' in result.stderr:
            conflicts = self.get_conflicting_files()
            return RebaseResult(
                success=False,
                conflicts=conflicts,
                message=f"Conflicts in {len(conflicts)} file(s)"
            )

        raise GitError(f"Rebase continue failed: {result.stderr or result.stdout}")

    def get_head_hash_quick(self) -> Optional[str]:
        """
        Get HEAD commit hash by reading .git/HEAD directly.

        This is faster than running a git subprocess for quick change detection.

        Returns:
            Short hash (8 chars) of HEAD commit, or None if unavailable
        """
        try:
            head_path = Path(self.working_dir) / '.git' / 'HEAD'
            if not head_path.exists():
                return None

            content = head_path.read_text().strip()

            if content.startswith('ref: '):
                # HEAD points to a branch ref
                ref_path = Path(self.working_dir) / '.git' / content[5:]
                if ref_path.exists():
                    return ref_path.read_text().strip()[:8]
                # Try packed-refs if ref file doesn't exist
                packed_refs = Path(self.working_dir) / '.git' / 'packed-refs'
                if packed_refs.exists():
                    ref_name = content[5:]
                    for line in packed_refs.read_text().splitlines():
                        if line.endswith(ref_name):
                            return line.split()[0][:8]
                return None
            else:
                # Detached HEAD - content is the hash
                return content[:8]
        except Exception:
            return None

    def resolve_conflict(self, file_path: str, keep: str) -> None:
        """
        Resolve a merge conflict by keeping one version.

        Args:
            file_path: Path to the conflicting file (relative to repo)
            keep: 'ours' to keep current branch version, 'theirs' for incoming
        """
        if keep == 'ours':
            self._run_git(['checkout', '--ours', file_path])
        elif keep == 'theirs':
            self._run_git(['checkout', '--theirs', file_path])
        else:
            raise ValueError(f"keep must be 'ours' or 'theirs', got '{keep}'")

        self._run_git(['add', file_path])

    def complete_merge(self, message: Optional[str] = None) -> None:
        """
        Complete a merge after all conflicts are resolved.

        Args:
            message: Optional commit message (uses default if not provided)
        """
        if message:
            self._run_git(['commit', '-m', message])
        else:
            # Use default merge message
            self._run_git(['commit', '--no-edit'])

    def abort_merge(self) -> None:
        """Abort an in-progress merge and restore previous state."""
        self._run_git(['merge', '--abort'])

    def get_file_from_ref(self, file_path: str, ref: str) -> Optional[str]:
        """
        Get file content from a specific git ref.

        Args:
            file_path: Path to file (relative to repo root)
            ref: Git ref (e.g., 'HEAD', 'MERGE_HEAD', branch name)

        Returns:
            File content as string, or None if not found
        """
        try:
            result = self._run_git(['show', f'{ref}:{file_path}'], check=False)
            if result.returncode == 0:
                return result.stdout
            return None
        except GitError:
            return None

    def get_file_hash(self, file_path: str) -> Optional[str]:
        """
        Get the git hash of a file for change detection.

        Args:
            file_path: Path to the file

        Returns:
            Git hash of the file contents, or None if file not found/not in repo
        """
        try:
            result = self._run_git(['hash-object', file_path], check=False)
            if result.returncode == 0:
                return result.stdout.strip()
            return None
        except (GitError, Exception):
            return None

    def get_log(self, limit: int = 10) -> List[CommitInfo]:
        """
        Get commit history.

        Args:
            limit: Maximum number of commits to return

        Returns:
            List of CommitInfo objects
        """
        format_str = '%H|%h|%an|%ad|%s'
        result = self._run_git([
            'log',
            f'-{limit}',
            f'--format={format_str}',
            '--date=short'
        ], check=False)

        commits = []
        if result.returncode == 0:
            for line in result.stdout.strip().split('\n'):
                if not line:
                    continue
                parts = line.split('|', 4)
                if len(parts) == 5:
                    commits.append(CommitInfo(
                        hash=parts[0],
                        short_hash=parts[1],
                        author=parts[2],
                        date=parts[3],
                        message=parts[4]
                    ))
        return commits

    def get_graph_log(self, limit: int = 30, skip: int = 0, all_branches: bool = True) -> List[GraphCommit]:
        """
        Get commit history with parent information for graph visualization.

        Args:
            limit: Maximum number of commits to return
            skip: Number of commits to skip (for pagination)
            all_branches: If True, include commits from all branches

        Returns:
            List of GraphCommit objects with parent hashes and refs
        """
        # Format: hash|short_hash|parents|refs|author|date|relative_date|message
        # %P gives parent hashes separated by spaces
        # %D gives refs (branches/tags) pointing to this commit
        format_str = '%H|%h|%P|%D|%an|%ad|%ar|%s'
        args = [
            'log',
            f'--skip={skip}',
            f'-{limit}',
            f'--format={format_str}',
            '--date=short'
        ]
        if all_branches:
            args.append('--all')

        result = self._run_git(args, check=False)

        commits = []
        if result.returncode == 0:
            for line in result.stdout.strip().split('\n'):
                if not line:
                    continue
                parts = line.split('|', 7)
                if len(parts) >= 8:
                    # Parse parents (space-separated hashes)
                    parents = parts[2].split() if parts[2] else []
                    # Parse refs (comma-separated, may include HEAD ->, origin/main, etc.)
                    refs = []
                    if parts[3]:
                        for ref in parts[3].split(', '):
                            ref = ref.strip()
                            # Skip HEAD -> references, just keep branch names
                            if ref.startswith('HEAD -> '):
                                ref = ref[8:]
                            if ref:
                                refs.append(ref)

                    commits.append(GraphCommit(
                        hash=parts[0],
                        short_hash=parts[1],
                        parents=parents,
                        refs=refs,
                        author=parts[4],
                        date=parts[5],
                        relative_date=parts[6],
                        message=parts[7]
                    ))
        return commits

    def get_all_branches(self) -> List[BranchInfo]:
        """
        Get all local and remote branches with their HEAD commits.

        Returns:
            List of BranchInfo objects
        """
        branches = []

        # Get all branches with their commit hashes
        # Format: hash refname is_current
        result = self._run_git([
            'for-each-ref',
            '--format=%(objectname:short)|%(refname:short)|%(refname)',
            'refs/heads',
            'refs/remotes'
        ], check=False)

        if result.returncode != 0:
            return branches

        # Get current branch
        current_result = self._run_git(['rev-parse', '--abbrev-ref', 'HEAD'], check=False)
        current_branch = current_result.stdout.strip() if current_result.returncode == 0 else ''

        for line in result.stdout.strip().split('\n'):
            if not line:
                continue
            parts = line.split('|', 2)
            if len(parts) >= 3:
                commit_hash = parts[0]
                short_name = parts[1]
                full_ref = parts[2]

                # Determine if remote
                is_remote = full_ref.startswith('refs/remotes/')

                # Skip HEAD entries from remotes
                if short_name.endswith('/HEAD'):
                    continue

                branches.append(BranchInfo(
                    name=short_name,
                    is_remote=is_remote,
                    is_current=(short_name == current_branch),
                    head_commit=commit_hash
                ))

        return branches

    def set_remote(self, url: str, name: str = 'origin') -> bool:
        """
        Set or update the remote repository URL.

        Args:
            url: Remote repository URL
            name: Remote name (default: origin)

        Returns:
            True if successful
        """
        # Check if remote exists
        result = self._run_git(['remote'], check=False)
        remotes = result.stdout.strip().split('\n') if result.stdout.strip() else []

        if name in remotes:
            self._run_git(['remote', 'set-url', name, url])
        else:
            self._run_git(['remote', 'add', name, url])
        return True

    def get_remote_url(self, name: str = 'origin') -> Optional[str]:
        """
        Get the URL for a remote.

        Args:
            name: Remote name (default: origin)

        Returns:
            Remote URL or None if not set
        """
        result = self._run_git(['remote', 'get-url', name], check=False)
        if result.returncode == 0:
            return result.stdout.strip()
        return None

    def get_remotes(self) -> List[Tuple[str, str]]:
        """
        Get list of all remotes with their URLs.

        Returns:
            List of (name, url) tuples
        """
        result = self._run_git(['remote', '-v'], check=False)
        remotes = {}

        if result.returncode == 0 and result.stdout.strip():
            for line in result.stdout.strip().split('\n'):
                if not line:
                    continue
                parts = line.split()
                if len(parts) >= 2:
                    name = parts[0]
                    url = parts[1]
                    # Only add each remote once (skip fetch/push duplicates)
                    if name not in remotes:
                        remotes[name] = url

        return list(remotes.items())

    def remove_remote(self, name: str) -> bool:
        """
        Remove a remote.

        Args:
            name: Remote name to remove

        Returns:
            True if successful
        """
        self._run_git(['remote', 'remove', name])
        return True

    def rename_remote(self, old_name: str, new_name: str) -> bool:
        """
        Rename a remote.

        Args:
            old_name: Current remote name
            new_name: New remote name

        Returns:
            True if successful
        """
        self._run_git(['remote', 'rename', old_name, new_name])
        return True

    @staticmethod
    def clone(url: str, target_dir: str, branch: Optional[str] = None) -> GitOperationResult:
        """
        Clone a repository to target directory.

        Args:
            url: Repository URL to clone (should include auth token if needed)
            target_dir: Directory to clone into
            branch: Optional branch to checkout after clone

        Returns:
            GitOperationResult with success status and SSL warning if applicable

        Raises:
            GitNotFoundError: If git is not installed
            GitError: If clone fails
        """
        if not GitUtils.is_git_available():
            raise GitNotFoundError("Git is not installed on this system.")

        args = ['git', 'clone', url, target_dir]
        if branch:
            args.extend(['-b', branch])

        # First try with SSL verification enabled
        try:
            result = subprocess.run(
                args,
                capture_output=True,
                text=True,
                timeout=300  # 5 minute timeout for clone
            )
            if result.returncode != 0:
                error_msg = result.stderr.strip() or result.stdout.strip()

                # If SSL error, retry without SSL verification
                if _is_ssl_error(error_msg):
                    args_no_ssl = ['git', '-c', 'http.sslVerify=false', 'clone', url, target_dir]
                    if branch:
                        args_no_ssl.extend(['-b', branch])

                    result2 = subprocess.run(
                        args_no_ssl,
                        capture_output=True,
                        text=True,
                        timeout=300
                    )
                    if result2.returncode != 0:
                        error_msg2 = result2.stderr.strip() or result2.stdout.strip()
                        raise GitError(f"Clone failed: {error_msg2}")

                    return GitOperationResult(
                        success=True,
                        ssl_verified=False,
                        warning="SSL certificate verification was disabled for this operation. "
                                "The connection may not be secure."
                    )

                raise GitError(f"Clone failed: {error_msg}")

            return GitOperationResult(success=True, ssl_verified=True)
        except subprocess.TimeoutExpired:
            raise GitError("Clone timed out")

    def configure_credential_helper(self, helper: str = 'store') -> bool:
        """
        Configure Git credential helper for the repository.

        Args:
            helper: Credential helper to use (store, cache, manager, etc.)

        Returns:
            True if successful
        """
        self._run_git(['config', 'credential.helper', helper])
        return True

    def set_config(self, key: str, value: str, global_config: bool = False) -> bool:
        """
        Set a Git configuration value.

        Args:
            key: Configuration key (e.g., 'user.name')
            value: Configuration value
            global_config: If True, sets global config; otherwise local

        Returns:
            True if successful
        """
        args = ['config']
        if global_config:
            args.append('--global')
        args.extend([key, value])
        self._run_git(args)
        return True
