"""
OAuth authentication handler for Paplix Version Control Plugin.

Implements OAuth 2.0 authorization code flow with PKCE for secure
authentication against the Paplix web service.
"""

import base64
import hashlib
import http.server
import json
import os
import secrets
import socketserver
import ssl
import threading
import urllib.parse
import webbrowser
from dataclasses import dataclass, field
from typing import Callable, Optional, Dict
from urllib.request import Request, urlopen
from urllib.error import URLError, HTTPError


# Module-level variable to track SSL warning
_ssl_warning: Optional[str] = None


def get_ssl_warning() -> Optional[str]:
    """Get the SSL warning message if SSL verification was disabled."""
    return _ssl_warning


def clear_ssl_warning():
    """Clear the SSL warning."""
    global _ssl_warning
    _ssl_warning = None


def _create_ssl_context(verified: bool = True):
    """Create SSL context.

    Args:
        verified: If True, create verified context. If False, create unverified.
    """
    if not verified:
        ctx = ssl.create_default_context()
        ctx.check_hostname = False
        ctx.verify_mode = ssl.CERT_NONE
        return ctx

    # Try certifi package first (if installed)
    try:
        import certifi
        return ssl.create_default_context(cafile=certifi.where())
    except ImportError:
        pass

    # Try default context with system certs
    return ssl.create_default_context()


def _urlopen_with_ssl_fallback(request: Request, timeout: int = 30):
    """
    Open URL with SSL fallback.

    First tries with SSL verification, falls back to unverified if SSL error.
    Sets module-level _ssl_warning if fallback was used.

    Returns:
        Response object
    """
    global _ssl_warning

    # First try with SSL verification
    try:
        return urlopen(request, context=_create_ssl_context(verified=True), timeout=timeout)
    except URLError as e:
        error_str = str(e)
        # Check if it's an SSL-related error
        ssl_indicators = ['SSL', 'ssl', 'certificate', 'CERTIFICATE', 'TLS', 'tls']
        if any(indicator in error_str for indicator in ssl_indicators):
            # Retry without SSL verification
            _ssl_warning = ("SSL certificate verification was disabled. "
                           "The connection may not be secure.")
            return urlopen(request, context=_create_ssl_context(verified=False), timeout=timeout)
        raise


@dataclass
class OAuthConfig:
    """OAuth configuration parameters."""
    client_id: str = "oa_client_kicad"
    client_secret: str = "oa_secret_kicad_dev_only_insecure"
    redirect_port: int = 8765
    scopes: str = "gitea:read gitea:write"


@dataclass
class OAuthDiscovery:
    """OAuth discovery document from .well-known endpoint."""
    issuer: str
    authorization_endpoint: str
    token_endpoint: str
    userinfo_endpoint: str
    revocation_endpoint: str
    gitea_url: str
    scopes_supported: list = field(default_factory=list)
    code_challenge_methods_supported: list = field(default_factory=list)


@dataclass
class OAuthTokens:
    """OAuth token response."""
    access_token: str
    refresh_token: Optional[str] = None
    token_type: str = "Bearer"
    expires_in: Optional[int] = None
    scope: Optional[str] = None


@dataclass
class UserInfo:
    """User information from userinfo endpoint."""
    gitea_access_token: str
    gitea_username: str
    gitea_org_name: str
    raw_data: Dict = field(default_factory=dict)


class OAuthError(Exception):
    """Exception raised for OAuth errors."""
    pass


class OAuthHandler:
    """
    Handles OAuth 2.0 authentication flow.

    Uses authorization code flow with PKCE (Proof Key for Code Exchange)
    for enhanced security. Discovers endpoints via .well-known endpoint.
    """

    DISCOVERY_PATH = "/.well-known/oauth-authorization-server"

    def __init__(self, base_url: str, config: Optional[OAuthConfig] = None):
        """
        Initialize OAuth handler.

        Args:
            base_url: Base URL of the Paplix service
            config: OAuth configuration (uses defaults if not provided)
        """
        self.base_url = base_url.rstrip('/')
        self.config = config or OAuthConfig()
        self._discovery: Optional[OAuthDiscovery] = None
        self._code_verifier: Optional[str] = None
        self._state: Optional[str] = None
        self._auth_code: Optional[str] = None
        self._error: Optional[str] = None
        self._server: Optional[socketserver.TCPServer] = None
        self._actual_port: Optional[int] = None

    @property
    def discovery_endpoint(self) -> str:
        """Get the discovery endpoint URL."""
        return f"{self.base_url}{self.DISCOVERY_PATH}"

    @property
    def redirect_uri(self) -> str:
        """Get the redirect URI for OAuth callback."""
        port = self._actual_port or self.config.redirect_port
        return f"http://localhost:{port}/callback"

    def discover(self) -> OAuthDiscovery:
        """
        Fetch OAuth configuration from discovery endpoint.

        Returns:
            OAuthDiscovery with all endpoint URLs

        Raises:
            OAuthError: If discovery fails
        """
        try:
            request = Request(
                self.discovery_endpoint,
                headers={'Accept': 'application/json'}
            )

            with _urlopen_with_ssl_fallback(request, timeout=30) as response:
                data = json.loads(response.read().decode())

            self._discovery = OAuthDiscovery(
                issuer=data['issuer'],
                authorization_endpoint=data['authorization_endpoint'],
                token_endpoint=data['token_endpoint'],
                userinfo_endpoint=data['userinfo_endpoint'],
                revocation_endpoint=data['revocation_endpoint'],
                gitea_url=data['gitea_url'],
                scopes_supported=data.get('scopes_supported', []),
                code_challenge_methods_supported=data.get('code_challenge_methods_supported', ['S256'])
            )
            return self._discovery
        except HTTPError as e:
            error_body = e.read().decode() if e.fp else ''
            raise OAuthError(f"Discovery failed: {e.code} - {error_body}")
        except URLError as e:
            raise OAuthError(f"Network error during discovery: {e.reason}")
        except (json.JSONDecodeError, KeyError) as e:
            raise OAuthError(f"Invalid discovery response: {e}")

    def _ensure_discovery(self) -> OAuthDiscovery:
        """Ensure discovery has been performed."""
        if self._discovery is None:
            return self.discover()
        return self._discovery

    def _generate_code_verifier(self) -> str:
        """Generate a cryptographically random code verifier for PKCE."""
        return base64.urlsafe_b64encode(secrets.token_bytes(32)).decode().rstrip('=')

    def _generate_code_challenge(self, verifier: str) -> str:
        """Generate code challenge from verifier using S256 method."""
        digest = hashlib.sha256(verifier.encode('ascii')).digest()
        return base64.urlsafe_b64encode(digest).rstrip(b'=').decode('ascii')

    def _generate_state(self) -> str:
        """Generate a random state parameter for CSRF protection."""
        return secrets.token_urlsafe(16)

    def get_authorization_url(self) -> str:
        """
        Build the authorization URL for the OAuth flow.

        Returns:
            URL to redirect user to for authorization

        Raises:
            OAuthError: If discovery fails
        """
        discovery = self._ensure_discovery()

        self._code_verifier = self._generate_code_verifier()
        self._state = self._generate_state()
        code_challenge = self._generate_code_challenge(self._code_verifier)

        params = {
            'client_id': self.config.client_id,
            'redirect_uri': self.redirect_uri,
            'response_type': 'code',
            'state': self._state,
            'code_challenge': code_challenge,
            'code_challenge_method': 'S256',
            'scope': self.config.scopes
        }

        query_string = urllib.parse.urlencode(params)
        return f"{discovery.authorization_endpoint}?{query_string}"

    def start_auth_flow(self, on_complete: Optional[Callable[[OAuthTokens], None]] = None,
                        on_error: Optional[Callable[[str], None]] = None) -> bool:
        """
        Start the OAuth authorization flow.

        Opens the browser for user authentication and starts a local
        server to receive the callback.

        Args:
            on_complete: Callback function called with tokens on success
            on_error: Callback function called with error message on failure

        Returns:
            True if flow was started successfully
        """
        # Create callback handler
        handler = self._create_callback_handler(on_complete, on_error)

        # Start local server to receive callback (port 0 = OS assigns available port)
        try:
            self._server = socketserver.TCPServer(
                ("localhost", 0),
                handler
            )
            # Get the actual port assigned by the OS
            self._actual_port = self._server.server_address[1]
            self._server.timeout = 300  # 5 minute timeout

            # Build auth URL after server starts (so redirect_uri has correct port)
            auth_url = self.get_authorization_url()

            # Run server in background thread
            server_thread = threading.Thread(target=self._server.handle_request)
            server_thread.daemon = True
            server_thread.start()

            # Open browser for authentication
            webbrowser.open(auth_url)

            return True
        except OSError as e:
            if on_error:
                on_error(f"Failed to start callback server: {e}")
            return False

    def _create_callback_handler(self, on_complete: Optional[Callable[[OAuthTokens], None]] = None,
                                  on_error: Optional[Callable[[str], None]] = None):
        """Create HTTP request handler for OAuth callback."""
        oauth_handler = self

        class CallbackHandler(http.server.BaseHTTPRequestHandler):
            def do_GET(self):
                """Handle OAuth callback GET request."""
                parsed = urllib.parse.urlparse(self.path)

                if parsed.path != '/callback':
                    self.send_error(404)
                    return

                query_params = urllib.parse.parse_qs(parsed.query)

                # Check for error
                if 'error' in query_params:
                    error = query_params['error'][0]
                    error_description = query_params.get('error_description', ['Unknown error'])[0]
                    self._send_error_response(f"{error}: {error_description}")
                    if on_error:
                        on_error(f"{error}: {error_description}")
                    return

                # Verify state
                received_state = query_params.get('state', [None])[0]
                if received_state != oauth_handler._state:
                    self._send_error_response("Invalid state parameter")
                    if on_error:
                        on_error("Invalid state parameter - possible CSRF attack")
                    return

                # Get authorization code
                code = query_params.get('code', [None])[0]
                if not code:
                    self._send_error_response("No authorization code received")
                    if on_error:
                        on_error("No authorization code received")
                    return

                # Exchange code for tokens
                try:
                    tokens = oauth_handler.exchange_code(code)
                    self._send_success_response()
                    if on_complete:
                        on_complete(tokens)
                except OAuthError as e:
                    self._send_error_response(str(e))
                    if on_error:
                        on_error(str(e))

            def _send_success_response(self):
                """Send success HTML response."""
                self.send_response(200)
                self.send_header('Content-type', 'text/html')
                self.end_headers()
                html = """
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Authentication Successful</title>
                    <style>
                        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                               display: flex; justify-content: center; align-items: center; height: 100vh;
                               margin: 0; background: #f5f5f5; }
                        .container { text-align: center; padding: 40px; background: white;
                                    border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
                        h1 { color: #22c55e; }
                        p { color: #666; }
                    </style>
                </head>
                <body>
                    <div class="container">
                        <h1>Authentication Successful!</h1>
                        <p>You can close this window and return to KiCad.</p>
                    </div>
                </body>
                </html>
                """
                self.wfile.write(html.encode())

            def _send_error_response(self, error: str):
                """Send error HTML response."""
                self.send_response(400)
                self.send_header('Content-type', 'text/html')
                self.end_headers()
                html = f"""
                <!DOCTYPE html>
                <html>
                <head>
                    <title>Authentication Failed</title>
                    <style>
                        body {{ font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
                               display: flex; justify-content: center; align-items: center; height: 100vh;
                               margin: 0; background: #f5f5f5; }}
                        .container {{ text-align: center; padding: 40px; background: white;
                                    border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }}
                        h1 {{ color: #ef4444; }}
                        p {{ color: #666; }}
                        .error {{ background: #fee2e2; padding: 10px; border-radius: 4px; margin-top: 20px; }}
                    </style>
                </head>
                <body>
                    <div class="container">
                        <h1>Authentication Failed</h1>
                        <p>There was an error during authentication.</p>
                        <div class="error">{error}</div>
                        <p>Please close this window and try again.</p>
                    </div>
                </body>
                </html>
                """
                self.wfile.write(html.encode())

            def log_message(self, format, *args):
                """Suppress server logging."""
                pass

        return CallbackHandler

    def exchange_code(self, code: str) -> OAuthTokens:
        """
        Exchange authorization code for access tokens.

        Args:
            code: Authorization code received from callback

        Returns:
            OAuthTokens with access and refresh tokens

        Raises:
            OAuthError: If token exchange fails
        """
        if not self._code_verifier:
            raise OAuthError("Code verifier not set - authorization flow not started")

        discovery = self._ensure_discovery()

        data = {
            'grant_type': 'authorization_code',
            'code': code,
            'redirect_uri': self.redirect_uri,
            'client_id': self.config.client_id,
            'client_secret': self.config.client_secret,
            'code_verifier': self._code_verifier
        }

        try:
            request = Request(
                discovery.token_endpoint,
                data=urllib.parse.urlencode(data).encode(),
                headers={
                    'Content-Type': 'application/x-www-form-urlencoded',
                    'Accept': 'application/json'
                }
            )

            with _urlopen_with_ssl_fallback(request, timeout=30) as response:
                response_data = json.loads(response.read().decode())

            return OAuthTokens(
                access_token=response_data['access_token'],
                refresh_token=response_data.get('refresh_token'),
                token_type=response_data.get('token_type', 'Bearer'),
                expires_in=response_data.get('expires_in'),
                scope=response_data.get('scope')
            )
        except HTTPError as e:
            error_body = e.read().decode() if e.fp else ''
            raise OAuthError(f"Token exchange failed: {e.code} - {error_body}")
        except URLError as e:
            raise OAuthError(f"Network error during token exchange: {e.reason}")
        except (json.JSONDecodeError, KeyError) as e:
            raise OAuthError(f"Invalid token response: {e}")

    def refresh_tokens(self, refresh_token: str) -> OAuthTokens:
        """
        Refresh access token using refresh token.

        Args:
            refresh_token: The refresh token

        Returns:
            New OAuthTokens

        Raises:
            OAuthError: If token refresh fails
        """
        discovery = self._ensure_discovery()

        data = {
            'grant_type': 'refresh_token',
            'client_id': self.config.client_id,
            'client_secret': self.config.client_secret,
            'refresh_token': refresh_token
        }

        try:
            request = Request(
                discovery.token_endpoint,
                data=urllib.parse.urlencode(data).encode(),
                headers={
                    'Content-Type': 'application/x-www-form-urlencoded',
                    'Accept': 'application/json'
                }
            )

            with _urlopen_with_ssl_fallback(request, timeout=30) as response:
                response_data = json.loads(response.read().decode())

            return OAuthTokens(
                access_token=response_data['access_token'],
                refresh_token=response_data.get('refresh_token', refresh_token),
                token_type=response_data.get('token_type', 'Bearer'),
                expires_in=response_data.get('expires_in'),
                scope=response_data.get('scope')
            )
        except HTTPError as e:
            error_body = e.read().decode() if e.fp else ''
            raise OAuthError(f"Token refresh failed: {e.code} - {error_body}")
        except URLError as e:
            raise OAuthError(f"Network error during token refresh: {e.reason}")
        except (json.JSONDecodeError, KeyError) as e:
            raise OAuthError(f"Invalid refresh response: {e}")

    def get_user_info(self, access_token: str) -> UserInfo:
        """
        Fetch user information using access token.

        Args:
            access_token: Valid access token

        Returns:
            UserInfo with Gitea credentials

        Raises:
            OAuthError: If fetching user info fails
        """
        discovery = self._ensure_discovery()

        try:
            request = Request(
                discovery.userinfo_endpoint,
                headers={
                    'Authorization': f'Bearer {access_token}',
                    'Accept': 'application/json'
                }
            )

            with _urlopen_with_ssl_fallback(request, timeout=30) as response:
                data = json.loads(response.read().decode())

            return UserInfo(
                gitea_access_token=data['gitea_access_token'],
                gitea_username=data['gitea_username'],
                gitea_org_name=data['gitea_org_name'],
                raw_data=data
            )
        except HTTPError as e:
            error_body = e.read().decode() if e.fp else ''
            raise OAuthError(f"Failed to fetch user info: {e.code} - {error_body}")
        except URLError as e:
            raise OAuthError(f"Network error fetching user info: {e.reason}")
        except (json.JSONDecodeError, KeyError) as e:
            raise OAuthError(f"Invalid user info response: {e}")

    def get_gitea_url(self) -> str:
        """
        Get the Gitea URL from discovery.

        Returns:
            Gitea URL for git operations
        """
        discovery = self._ensure_discovery()
        return discovery.gitea_url

    def shutdown(self):
        """Shutdown the callback server if running."""
        if self._server:
            self._server.shutdown()
            self._server = None
