"""
Paplix Version Control Action Plugin for KiCad.

Main entry point for the KiCad plugin that provides version control
functionality through a simplified Git interface.
"""

import os
import pcbnew
import wx

from .config import Config
from .git_utils import GitUtils
from .dialogs import VersionControlPanel, GitNotInstalledDialog, SetupDialog


class PaplixVCPlugin(pcbnew.ActionPlugin):
    """
    KiCad Action Plugin for Paplix Version Control.

    Provides a simplified interface for version control operations
    designed for engineers without extensive Git knowledge.
    """

    # Class-level panel reference for singleton pattern
    _panel = None
    _current_project = None

    @staticmethod
    def _save_current_board():
        """
        Save the current KiCad board file.

        Returns:
            Tuple of (success: bool, error_message: Optional[str])
        """
        try:
            board = pcbnew.GetBoard()
            if board:
                board_path = board.GetFileName()
                if board_path:
                    pcbnew.SaveBoard(board_path, board)
                    return True, None
        except Exception as e:
            return False, str(e)
        return False, "No board open"

    def defaults(self):
        """Set plugin defaults for KiCad."""
        self.name = "Paplix Version Control"
        self.category = "Version Control"
        self.description = (
            "Simplified version control for KiCad projects. "
            "Commit, push, pull, and manage branches without Git expertise."
        )
        self.show_toolbar_button = True

        # Set icon path - check multiple possible locations
        plugin_dir = os.path.dirname(os.path.abspath(__file__))

        # Possible icon locations:
        # 1. Development: plugin_dir/../resources/icon.png
        # 2. PCM installed: plugin_dir/../../resources/com_paplix_kicad-vc/icon.png
        icon_paths = [
            os.path.join(plugin_dir, '..', 'resources', 'icon.png'),
            os.path.join(plugin_dir, '..', '..', 'resources', 'com_paplix_kicad-vc', 'icon.png'),
        ]

        for path in icon_paths:
            if os.path.exists(path):
                self.icon_file_name = os.path.abspath(path)
                break

        dark_icon_paths = [
            os.path.join(plugin_dir, '..', 'resources', 'icon_dark.png'),
            os.path.join(plugin_dir, '..', '..', 'resources', 'com_paplix_kicad-vc', 'icon_dark.png'),
        ]

        for path in dark_icon_paths:
            if os.path.exists(path):
                self.dark_icon_file_name = os.path.abspath(path)
                break

    def Run(self):
        """Execute the plugin when invoked from KiCad."""
        # Check if Git is installed
        if not GitUtils.is_git_available():
            dlg = GitNotInstalledDialog(None)
            dlg.ShowModal()
            dlg.Destroy()
            return

        # Get the current board and project path
        board = pcbnew.GetBoard()
        if board is None:
            wx.MessageBox(
                "No board is currently open.\n"
                "Please open a PCB file first.",
                "No Board Open",
                wx.OK | wx.ICON_WARNING
            )
            return

        board_path = board.GetFileName()
        if not board_path:
            wx.MessageBox(
                "The current board has not been saved.\n"
                "Please save your project first.",
                "Board Not Saved",
                wx.OK | wx.ICON_WARNING
            )
            return

        # Get project directory
        project_path = os.path.dirname(board_path)

        # Check if this is the first run (no URL configured)
        config = Config()
        if not config.base_url:
            # Show setup dialog
            setup_dlg = SetupDialog(None, config)
            result = setup_dlg.ShowModal()
            setup_dlg.Destroy()

            if result != wx.ID_SAVE:
                # User cancelled setup
                return

        # Use singleton pattern for the panel
        # If project changed, recreate the panel
        if PaplixVCPlugin._panel is not None and PaplixVCPlugin._current_project != project_path:
            try:
                PaplixVCPlugin._panel.Destroy()
            except:
                pass
            PaplixVCPlugin._panel = None

        # Create panel if it doesn't exist
        if PaplixVCPlugin._panel is None:
            PaplixVCPlugin._panel = VersionControlPanel(None, project_path, config)
            PaplixVCPlugin._current_project = project_path

        # Show or raise the panel
        if PaplixVCPlugin._panel.IsShown():
            PaplixVCPlugin._panel.Raise()
            PaplixVCPlugin._panel._refresh_status()
        else:
            PaplixVCPlugin._panel.Show()
            PaplixVCPlugin._panel._refresh_status()

            # Run auto-operations on show
            PaplixVCPlugin._run_auto_operations(PaplixVCPlugin._panel)

    @staticmethod
    def _run_auto_operations(panel):
        """Run auto-pull and auto-save operations."""
        status = panel.git.get_status()
        if not status.is_repo:
            return

        # Auto-pull
        if panel.project_config.auto_pull_on_open and status.has_remote and status.has_upstream:
            try:
                result = panel.git.pull()
                if result.warning:
                    panel._show_status("Auto-pull completed (SSL warning)", "warning")
                else:
                    panel._show_status("Auto-pull completed", "success")
            except Exception as e:
                panel._show_status(f"Auto-pull failed", "error")
                return

        # Refresh status after pull
        status = panel.git.get_status()

        # Auto-save (save files only - no commit)
        if panel.project_config.auto_commit_on_focus:
            try:
                success, error = PaplixVCPlugin._save_current_board()
                if success:
                    panel._show_status("Files saved", "success")
                else:
                    panel._show_status(f"Auto-save failed: {error}", "error")
            except Exception as e:
                panel._show_status("Auto-save failed", "error")

        panel._refresh_status()
